@props(['fallback' => null])

<div x-data="errorBoundary()" x-init="init()">
    <template x-if="!hasError">
        <div>
            {{ $slot }}
        </div>
    </template>
    
    <template x-if="hasError">
        <div class="bg-red-50 border border-red-200 rounded-lg p-6">
            <div class="flex">
                <div class="flex-shrink-0">
                    <svg class="h-5 w-5 text-red-400" fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"></path>
                    </svg>
                </div>
                <div class="ml-3">
                    <h3 class="text-sm font-medium text-red-800">Something went wrong</h3>
                    <div class="mt-2 text-sm text-red-700">
                        <p>An error occurred while loading this component. Please try refreshing the page.</p>
                        @if($fallback)
                            <div class="mt-4">
                                {{ $fallback }}
                            </div>
                        @endif
                    </div>
                    <div class="mt-4">
                        <div class="-mx-2 -my-1.5 flex">
                            <button type="button" 
                                    @click="retry()"
                                    class="bg-red-50 px-2 py-1.5 rounded-md text-sm font-medium text-red-800 hover:bg-red-100 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-offset-red-50 focus:ring-red-600">
                                Try Again
                            </button>
                            <button type="button" 
                                    @click="reload()"
                                    class="ml-3 bg-red-50 px-2 py-1.5 rounded-md text-sm font-medium text-red-800 hover:bg-red-100 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-offset-red-50 focus:ring-red-600">
                                Reload Page
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </template>
</div>

<script>
function errorBoundary() {
    return {
        hasError: false,
        error: null,
        
        init() {
            // Listen for Livewire errors
            document.addEventListener('livewire:exception', (event) => {
                this.handleError(event.detail);
            });
            
            // Listen for general JavaScript errors
            window.addEventListener('error', (event) => {
                this.handleError(event);
            });
            
            // Listen for unhandled promise rejections
            window.addEventListener('unhandledrejection', (event) => {
                this.handleError(event);
            });
        },
        
        handleError(error) {
            console.error('Error caught by boundary:', error);
            this.hasError = true;
            this.error = error;
            
            // Log error to server if possible
            this.logError(error);
        },
        
        retry() {
            this.hasError = false;
            this.error = null;
            
            // Trigger Livewire refresh if available
            if (window.Livewire) {
                window.Livewire.restart();
            }
        },
        
        reload() {
            window.location.reload();
        },
        
        logError(error) {
            // Send error to server for logging
            fetch('/api/log-error', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    error: error.message || 'Unknown error',
                    stack: error.stack || 'No stack trace',
                    url: window.location.href,
                    userAgent: navigator.userAgent,
                    timestamp: new Date().toISOString()
                })
            }).catch(err => {
                console.error('Failed to log error:', err);
            });
        }
    }
}
</script>
