@extends('components.layouts.admin')

@section('title', 'Edit Virtual Card')

@section('content')
<div class="space-y-6">
    <!-- Success Message -->
    @if (session('success'))
        <x-admin.message type="success">
            {{ session('success') }}
        </x-admin.message>
    @endif

    <!-- Error Message -->
    @if (session('error'))
        <x-admin.message type="error">
            {{ session('error') }}
        </x-admin.message>
    @endif

    <!-- Header -->
    <div class="flex justify-between items-center">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Edit Virtual Card</h1>
            <p class="text-gray-600">Update virtual card information and settings</p>
        </div>
        <div class="flex space-x-3">
            <flux:button href="{{ route('admin.virtual-cards.index') }}" variant="outline" class="text-xs">
                <flux:icon name="arrow-left" class="w-4 h-4 mr-2" />
                Back to Cards
            </flux:button>
            <flux:button href="{{ route('admin.virtual-cards.show', $virtualCard) }}" variant="outline" class="text-xs">
                <flux:icon name="eye" class="w-4 h-4 mr-2" />
                View Card
            </flux:button>
        </div>
    </div>

    <form method="POST" action="{{ route('admin.virtual-cards.update', $virtualCard) }}" class="space-y-6">
        @csrf
        @method('PUT')
        
        <!-- User and Account Selection -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">User and Account</flux:heading>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <flux:field>
                    <flux:label for="user_id">User</flux:label>
                    <flux:select id="user_id" name="user_id" required>
                        <option value="">Select a user</option>
                        @foreach($users as $user)
                            <option value="{{ $user->id }}" {{ old('user_id', $virtualCard->user_id) == $user->id ? 'selected' : '' }}>
                                {{ $user->name }} ({{ $user->email }})
                            </option>
                        @endforeach
                    </flux:select>
                    <flux:error name="user_id" />
                </flux:field>

                <flux:field>
                    <flux:label for="account_id">Account</flux:label>
                    <flux:select id="account_id" name="account_id" required>
                        <option value="">Select an account</option>
                        @foreach($accounts as $account)
                            <option value="{{ $account->id }}" {{ old('account_id', $virtualCard->account_id) == $account->id ? 'selected' : '' }}>
                                {{ $account->account_number }} - {{ $account->user->name }} ({{ $account->currency }})
                            </option>
                        @endforeach
                    </flux:select>
                    <flux:error name="account_id" />
                </flux:field>
            </div>
        </flux:card>

        <!-- Card Details -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Card Details</flux:heading>
            
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <flux:field>
                    <flux:label for="card_number">Card Number</flux:label>
                    <flux:input 
                        id="card_number" 
                        name="card_number" 
                        value="{{ old('card_number', $virtualCard->card_number) }}"
                        placeholder="1234567890123456"
                        class="font-mono"
                    />
                    <flux:error name="card_number" />
                    <flux:description>16-digit card number (or 15 for Amex)</flux:description>
                </flux:field>

                <flux:field>
                    <flux:label for="cvv">CVV</flux:label>
                    <flux:input 
                        id="cvv" 
                        name="cvv" 
                        value="{{ old('cvv', $virtualCard->cvv) }}"
                        placeholder="123"
                        maxlength="4"
                        class="font-mono"
                    />
                    <flux:error name="cvv" />
                    <flux:description>3-4 digit security code</flux:description>
                </flux:field>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
                <flux:field>
                    <flux:label for="card_type">Card Type</flux:label>
                    <flux:select id="card_type" name="card_type" required>
                        <option value="">Select card type</option>
                        <option value="debit" {{ old('card_type', $virtualCard->card_type) === 'debit' ? 'selected' : '' }}>Debit</option>
                        <option value="credit" {{ old('card_type', $virtualCard->card_type) === 'credit' ? 'selected' : '' }}>Credit</option>
                        <option value="prepaid" {{ old('card_type', $virtualCard->card_type) === 'prepaid' ? 'selected' : '' }}>Prepaid</option>
                    </flux:select>
                    <flux:error name="card_type" />
                </flux:field>

                <flux:field>
                    <flux:label for="card_brand">Card Brand</flux:label>
                    <flux:select id="card_brand" name="card_brand" required>
                        <option value="">Select card brand</option>
                        <option value="visa" {{ old('card_brand', $virtualCard->card_brand) === 'visa' ? 'selected' : '' }}>Visa</option>
                        <option value="mastercard" {{ old('card_brand', $virtualCard->card_brand) === 'mastercard' ? 'selected' : '' }}>Mastercard</option>
                        <option value="amex" {{ old('card_brand', $virtualCard->card_brand) === 'amex' ? 'selected' : '' }}>American Express</option>
                    </flux:select>
                    <flux:error name="card_brand" />
                </flux:field>

                <flux:field>
                    <flux:label for="card_name">Card Name (Optional)</flux:label>
                    <flux:input 
                        id="card_name" 
                        name="card_name" 
                        value="{{ old('card_name', $virtualCard->card_name) }}"
                        placeholder="e.g., My Shopping Card"
                    />
                    <flux:error name="card_name" />
                </flux:field>

                <flux:field>
                    <flux:label for="expiry_date">Expiry Date</flux:label>
                    <flux:date-picker 
                        id="expiry_date" 
                        name="expiry_date" 
                        value="{{ old('expiry_date', $virtualCard->expiry_date->format('Y-m-d')) }}"
                        required
                    />
                    <flux:error name="expiry_date" />
                </flux:field>

                <flux:field>
                    <flux:label for="currency">Currency</flux:label>
                    <flux:select id="currency" name="currency" required>
                        @foreach(\App\Helpers\CurrencyHelper::getSupportedCurrencies() as $code => $currency)
                            <option value="{{ $code }}" {{ old('currency', $virtualCard->currency) === $code ? 'selected' : '' }}>
                                {{ $code }} - {{ $currency['name'] }} ({{ $currency['symbol'] }})
                            </option>
                        @endforeach
                    </flux:select>
                    <flux:error name="currency" />
                </flux:field>
            </div>
        </flux:card>

        <!-- Financial Limits -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Financial Limits</flux:heading>
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                <flux:field>
                    <flux:label for="credit_limit">Credit Limit (Optional)</flux:label>
                    <flux:input 
                        id="credit_limit" 
                        name="credit_limit" 
                        type="number"
                        step="0.01"
                        min="0"
                        value="{{ old('credit_limit', $virtualCard->credit_limit) }}"
                        placeholder="0.00"
                    />
                    <flux:error name="credit_limit" />
                    <flux:description>For credit cards only</flux:description>
                </flux:field>

                <flux:field>
                    <flux:label for="daily_limit">Daily Limit (Optional)</flux:label>
                    <flux:input 
                        id="daily_limit" 
                        name="daily_limit" 
                        type="number"
                        step="0.01"
                        min="0"
                        value="{{ old('daily_limit', $virtualCard->daily_limit) }}"
                        placeholder="0.00"
                    />
                    <flux:error name="daily_limit" />
                </flux:field>

                <flux:field>
                    <flux:label for="monthly_limit">Monthly Limit (Optional)</flux:label>
                    <flux:input 
                        id="monthly_limit" 
                        name="monthly_limit" 
                        type="number"
                        step="0.01"
                        min="0"
                        value="{{ old('monthly_limit', $virtualCard->monthly_limit) }}"
                        placeholder="0.00"
                    />
                    <flux:error name="monthly_limit" />
                </flux:field>
            </div>
        </flux:card>

        <!-- Current Balance and Financial Info -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Current Balance & Financial Information</flux:heading>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <flux:field>
                    <flux:label for="current_balance">Current Balance</flux:label>
                    <div class="mb-2 p-3 bg-blue-50 rounded-lg border border-blue-200">
                        <div class="text-sm text-blue-600 mb-1">Current Formatted Balance</div>
                        <div class="text-lg font-semibold text-blue-800">{{ $virtualCard->formatted_current_balance }}</div>
                    </div>
                    <flux:input 
                        id="current_balance" 
                        name="current_balance" 
                        type="number"
                        step="0.01"
                        min="0"
                        value="{{ old('current_balance', $virtualCard->current_balance) }}"
                        placeholder="0.00"
                    />
                    <flux:error name="current_balance" />
                    <flux:description>Current available balance on the card (editable)</flux:description>
                </flux:field>

                <flux:field>
                    <flux:label for="total_spent">Total Spent (Read Only)</flux:label>
                    <div class="mb-2 p-3 bg-gray-50 rounded-lg border border-gray-200">
                        <div class="text-sm text-gray-600 mb-1">Formatted Total Spent</div>
                        <div class="text-lg font-semibold text-gray-800">{{ \App\Helpers\CurrencyHelper::getSymbol($virtualCard->currency) }}{{ number_format($virtualCard->total_spent, 2) }}</div>
                    </div>
                    <flux:input 
                        id="total_spent" 
                        name="total_spent" 
                        type="number"
                        step="0.01"
                        value="{{ $virtualCard->total_spent }}"
                        readonly
                        class="bg-gray-50"
                    />
                    <flux:description>Total amount spent using this card (calculated automatically)</flux:description>
                </flux:field>

                <flux:field>
                    <flux:label for="total_transactions">Total Transactions (Read Only)</flux:label>
                    <flux:input 
                        id="total_transactions" 
                        name="total_transactions" 
                        type="number"
                        value="{{ $virtualCard->total_transactions }}"
                        readonly
                        class="bg-gray-50"
                    />
                    <flux:description>Total number of transactions (calculated automatically)</flux:description>
                </flux:field>

                <flux:field>
                    <flux:label for="last_used_at">Last Used Date (Read Only)</flux:label>
                    <flux:input 
                        id="last_used_at" 
                        name="last_used_at" 
                        type="datetime-local"
                        value="{{ $virtualCard->last_used_at ? $virtualCard->last_used_at->format('Y-m-d\TH:i') : '' }}"
                        readonly
                        class="bg-gray-50"
                    />
                    <flux:description>Date when the card was last used for a transaction</flux:description>
                </flux:field>
            </div>
        </flux:card>

        <!-- Card Features -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Card Features</flux:heading>
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                <flux:field>
                    <flux:label for="online_payments">Online Payments</flux:label>
                    <flux:select id="online_payments" name="online_payments">
                        <option value="1" {{ old('online_payments', $virtualCard->online_payments ? '1' : '0') === '1' ? 'selected' : '' }}>Enabled</option>
                        <option value="0" {{ old('online_payments', $virtualCard->online_payments ? '1' : '0') === '0' ? 'selected' : '' }}>Disabled</option>
                    </flux:select>
                    <flux:error name="online_payments" />
                </flux:field>

                <flux:field>
                    <flux:label for="atm_withdrawals">ATM Withdrawals</flux:label>
                    <flux:select id="atm_withdrawals" name="atm_withdrawals">
                        <option value="1" {{ old('atm_withdrawals', $virtualCard->atm_withdrawals ? '1' : '0') === '1' ? 'selected' : '' }}>Enabled</option>
                        <option value="0" {{ old('atm_withdrawals', $virtualCard->atm_withdrawals ? '1' : '0') === '0' ? 'selected' : '' }}>Disabled</option>
                    </flux:select>
                    <flux:error name="atm_withdrawals" />
                </flux:field>

                <flux:field>
                    <flux:label for="international_usage">International Usage</flux:label>
                    <flux:select id="international_usage" name="international_usage">
                        <option value="0" {{ old('international_usage', $virtualCard->international_usage ? '1' : '0') === '0' ? 'selected' : '' }}>Disabled</option>
                        <option value="1" {{ old('international_usage', $virtualCard->international_usage ? '1' : '0') === '1' ? 'selected' : '' }}>Enabled</option>
                    </flux:select>
                    <flux:error name="international_usage" />
                </flux:field>
            </div>
        </flux:card>
        
        <!-- Actions -->
        <div class="flex justify-end space-x-3">
            <flux:button href="{{ route('admin.virtual-cards.show', $virtualCard) }}" variant="outline">
                Cancel
            </flux:button>
            <flux:button icon="check" type="submit" class="text-xs">
                Update Virtual Card
            </flux:button>
        </div>
    </form>
</div>

<script>
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(function() {
        // Show a temporary success message
        const button = event.target.closest('button');
        const originalText = button.innerHTML;
        button.innerHTML = '<flux:icon name="check" class="w-4 h-4 mr-1" />Copied!';
        button.classList.add('bg-green-100', 'text-green-800');
        
        setTimeout(() => {
            button.innerHTML = originalText;
            button.classList.remove('bg-green-100', 'text-green-800');
        }, 2000);
    }).catch(function(err) {
        console.error('Could not copy text: ', err);
        alert('Failed to copy card number to clipboard');
    });
}
</script>
@endsection
