@extends('components.layouts.admin')

@section('title', 'Transactions')

@section('content')
    <div class="space-y-6">
        <!-- Success Message -->
        @if (session('success'))
            <x-admin.message type="success">
                {{ session('success') }}
            </x-admin.message>
        @endif

        <!-- Error Message -->
        @if (session('error'))
            <x-admin.message type="error">
                {{ session('error') }}
            </x-admin.message>
        @endif

        <!-- Header -->
        <div class="flex justify-between items-center">
            <div>
                <h1 class="text-2xl font-bold text-gray-900">Transactions</h1>
                <p class="text-gray-600">View all banking transactions and their status</p>
            </div>
            <flux:button icon="plus" href="{{ route('admin.transactions.create') }}" variant="primary">
                Create Transaction
            </flux:button>
        </div>

        <!-- Filters -->
        <div class="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
            <form method="GET" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <!-- Search -->
                <flux:field>
                    <flux:label>Search</flux:label>
                    <flux:input 
                        name="search" 
                        value="{{ request('search') }}" 
                        placeholder="Transaction ID, reference, description..."
                    />
                </flux:field>

                <!-- Type -->
                <flux:field>
                    <flux:label>Type</flux:label>
                    <flux:select name="type">
                        <option value="">All Types</option>
                        <option value="deposit" {{ request('type') === 'deposit' ? 'selected' : '' }}>Deposit</option>
                        <option value="withdrawal" {{ request('type') === 'withdrawal' ? 'selected' : '' }}>Withdrawal</option>
                        <option value="transfer" {{ request('type') === 'transfer' ? 'selected' : '' }}>Transfer</option>
                        <option value="payment" {{ request('type') === 'payment' ? 'selected' : '' }}>Payment</option>
                        <option value="fee" {{ request('type') === 'fee' ? 'selected' : '' }}>Fee</option>
                        <option value="interest" {{ request('type') === 'interest' ? 'selected' : '' }}>Interest</option>
                        <option value="refund" {{ request('type') === 'refund' ? 'selected' : '' }}>Refund</option>
                        <option value="exchange" {{ request('type') === 'exchange' ? 'selected' : '' }}>Exchange</option>
                    </flux:select>
                </flux:field>

                <!-- Status -->
                <flux:field>
                    <flux:label>Status</flux:label>
                    <flux:select name="status">
                        <option value="">All Statuses</option>
                        <option value="pending" {{ request('status') === 'pending' ? 'selected' : '' }}>Pending</option>
                        <option value="processing" {{ request('status') === 'processing' ? 'selected' : '' }}>Processing</option>
                        <option value="completed" {{ request('status') === 'completed' ? 'selected' : '' }}>Completed</option>
                        <option value="failed" {{ request('status') === 'failed' ? 'selected' : '' }}>Failed</option>
                        <option value="cancelled" {{ request('status') === 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                        <option value="reversed" {{ request('status') === 'reversed' ? 'selected' : '' }}>Reversed</option>
                    </flux:select>
                </flux:field>

                <!-- Category -->
                <flux:field>
                    <flux:label>Category</flux:label>
                    <flux:select name="category">
                        <option value="">All Categories</option>
                        <option value="banking" {{ request('category') === 'banking' ? 'selected' : '' }}>Banking</option>
                        <option value="card" {{ request('category') === 'card' ? 'selected' : '' }}>Card</option>
                        <option value="investment" {{ request('category') === 'investment' ? 'selected' : '' }}>Investment</option>
                        <option value="loan" {{ request('category') === 'loan' ? 'selected' : '' }}>Loan</option>
                        <option value="insurance" {{ request('category') === 'insurance' ? 'selected' : '' }}>Insurance</option>
                        <option value="utility" {{ request('category') === 'utility' ? 'selected' : '' }}>Utility</option>
                    </flux:select>
                </flux:field>

                <!-- Currency -->
                <flux:field>
                    <flux:label>Currency</flux:label>
                    <flux:select name="currency">
                        <option value="">All Currencies</option>
                        <option value="USD" {{ request('currency') === 'USD' ? 'selected' : '' }}>USD</option>
                        <option value="EUR" {{ request('currency') === 'EUR' ? 'selected' : '' }}>EUR</option>
                        <option value="GBP" {{ request('currency') === 'GBP' ? 'selected' : '' }}>GBP</option>
                        <option value="NGN" {{ request('currency') === 'NGN' ? 'selected' : '' }}>NGN</option>
                    </flux:select>
                </flux:field>

                <!-- User -->
                <flux:field>
                    <flux:label>User</flux:label>
                    <flux:select name="user_id">
                        <option value="">All Users</option>
                        @foreach($users as $user)
                            <option value="{{ $user->id }}" {{ request('user_id') == $user->id ? 'selected' : '' }}>
                                {{ $user->name }} ({{ $user->email }})
                            </option>
                        @endforeach
                    </flux:select>
                </flux:field>

                <!-- From Account -->
                <flux:field>
                    <flux:label>From Account</flux:label>
                    <flux:select name="from_account_id">
                        <option value="">All Accounts</option>
                        @foreach($accounts as $account)
                            <option value="{{ $account->id }}" {{ request('from_account_id') == $account->id ? 'selected' : '' }}>
                                {{ $account->account_number }} - {{ $account->account_name }}
                            </option>
                        @endforeach
                    </flux:select>
                </flux:field>

                <!-- To Account -->
                <flux:field>
                    <flux:label>To Account</flux:label>
                    <flux:select name="to_account_id">
                        <option value="">All Accounts</option>
                        @foreach($accounts as $account)
                            <option value="{{ $account->id }}" {{ request('to_account_id') == $account->id ? 'selected' : '' }}>
                                {{ $account->account_number }} - {{ $account->account_name }}
                            </option>
                        @endforeach
                    </flux:select>
                </flux:field>

                <!-- Date From -->
                <flux:field>
                    <flux:label>Date From</flux:label>
                    <flux:date-picker 
                        name="date_from" 
                        value="{{ request('date_from') }}"
                    />
                </flux:field>

                <!-- Date To -->
                <flux:field>
                    <flux:label>Date To</flux:label>
                    <flux:date-picker 
                        name="date_to" 
                        value="{{ request('date_to') }}"
                    />
                </flux:field>

                <!-- Amount Min -->
                <flux:field>
                    <flux:label>Min Amount</flux:label>
                    <flux:input 
                        name="amount_min" 
                        type="number"
                        step="0.01"
                        value="{{ request('amount_min') }}" 
                        placeholder="0.00"
                    />
                </flux:field>

                <!-- Amount Max -->
                <flux:field>
                    <flux:label>Max Amount</flux:label>
                    <flux:input 
                        name="amount_max" 
                        type="number"
                        step="0.01"
                        value="{{ request('amount_max') }}" 
                        placeholder="0.00"
                    />
                </flux:field>

                <!-- Sort By -->
                <flux:field>
                    <flux:label>Sort By</flux:label>
                    <flux:select name="sort_by">
                        <option value="created_at" {{ request('sort_by') === 'created_at' ? 'selected' : '' }}>Created Date</option>
                        <option value="transaction_id" {{ request('sort_by') === 'transaction_id' ? 'selected' : '' }}>Transaction ID</option>
                        <option value="amount" {{ request('sort_by') === 'amount' ? 'selected' : '' }}>Amount</option>
                        <option value="status" {{ request('sort_by') === 'status' ? 'selected' : '' }}>Status</option>
                        <option value="type" {{ request('sort_by') === 'type' ? 'selected' : '' }}>Type</option>
                        <option value="processed_at" {{ request('sort_by') === 'processed_at' ? 'selected' : '' }}>Processed Date</option>
                        <option value="completed_at" {{ request('sort_by') === 'completed_at' ? 'selected' : '' }}>Completed Date</option>
                    </flux:select>
                </flux:field>

                <!-- Sort Direction -->
                <flux:field>
                    <flux:label>Sort Direction</flux:label>
                    <flux:select name="sort_direction">
                        <option value="desc" {{ request('sort_direction') === 'desc' ? 'selected' : '' }}>Descending</option>
                        <option value="asc" {{ request('sort_direction') === 'asc' ? 'selected' : '' }}>Ascending</option>
                    </flux:select>
                </flux:field>

                <!-- Filter Buttons -->
                <div class="flex gap-2">
                    <flux:button icon="magnifying-glass" type="submit" variant="primary">
                        Filter
                    </flux:button>
                    <flux:button icon="x-mark" href="{{ route('admin.transactions.index') }}" variant="outline">
                        Clear
                    </flux:button>
                </div>
            </form>
        </div>

        <!-- Transactions Table -->
        <div class="bg-white rounded-xl border border-gray-200 shadow-sm overflow-hidden">
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Transaction
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                User
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Type
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Amount
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Status
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Date
                            </th>
                            <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Actions
                            </th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        @forelse($transactions as $transaction)
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex items-center">
                                        <div>
                                            <div class="text-sm font-medium text-gray-900">
                                                {{ $transaction->transaction_id }}
                                            </div>
                                            <div class="text-sm text-gray-500">
                                                {{ $transaction->reference_number }}
                                            </div>
                                            @if($transaction->external_reference)
                                                <div class="text-xs text-gray-400">
                                                    Ext: {{ $transaction->external_reference }}
                                                </div>
                                            @endif
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-900">
                                        {{ $transaction->user->name }}
                                    </div>
                                    <div class="text-sm text-gray-500">
                                        {{ $transaction->user->email }}
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                        {{ ucfirst($transaction->type) }}
                                    </span>
                                    <div class="text-sm text-gray-500 mt-1">
                                        {{ ucfirst($transaction->category) }}
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm font-medium text-gray-900">
                                        {{ $transaction->currency }}{{ number_format($transaction->amount, 2) }}
                                    </div>
                                    @if($transaction->fee_amount > 0)
                                        <div class="text-sm text-gray-500">
                                            Fee: {{ $transaction->currency }}{{ number_format($transaction->fee_amount, 2) }}
                                        </div>
                                    @endif
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium
                                        @if($transaction->status === 'completed') bg-green-100 text-green-800
                                        @elseif($transaction->status === 'pending') bg-yellow-100 text-yellow-800
                                        @elseif($transaction->status === 'processing') bg-blue-100 text-blue-800
                                        @elseif($transaction->status === 'failed') bg-red-100 text-red-800
                                        @elseif($transaction->status === 'cancelled') bg-gray-100 text-gray-800
                                        @elseif($transaction->status === 'reversed') bg-purple-100 text-purple-800
                                        @else bg-gray-100 text-gray-800
                                        @endif">
                                        {{ ucfirst($transaction->status) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                    {{ $transaction->created_at->format('M d, Y') }}
                                    <div class="text-xs text-gray-400">
                                        {{ $transaction->created_at->format('g:i A') }}
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                    <div class="flex items-center justify-end space-x-2">
                                        <flux:button 
                                            href="{{ route('admin.transactions.show', $transaction) }}" 
                                            variant="outline" 
                                            size="sm"
                                        >
                                            <flux:icon name="eye" class="w-4 h-4" />
                                        </flux:button>
                                        <flux:button 
                                            href="{{ route('admin.transactions.edit', $transaction) }}" 
                                            variant="outline" 
                                            size="sm"
                                        >
                                            <flux:icon name="pencil" class="w-4 h-4" />
                                        </flux:button>
                                        
                                        @if($transaction->status === 'pending')
                                            <form method="POST" action="{{ route('admin.transactions.approve', $transaction) }}" class="inline">
                                                @csrf
                                                <flux:button type="submit" variant="outline" size="sm" class="text-green-600 hover:text-green-700">
                                                    <flux:icon name="check" class="w-4 h-4" />
                                                </flux:button>
                                            </form>
                                            <form method="POST" action="{{ route('admin.transactions.reject', $transaction) }}" class="inline">
                                                @csrf
                                                <flux:button type="submit" variant="outline" size="sm" class="text-red-600 hover:text-red-700">
                                                    <flux:icon name="x-mark" class="w-4 h-4" />
                                                </flux:button>
                                            </form>
                                        @endif

                                        @if($transaction->status === 'completed')
                                            <form method="POST" action="{{ route('admin.transactions.reverse', $transaction) }}" class="inline">
                                                @csrf
                                                <flux:button type="submit" variant="outline" size="sm" class="text-purple-600 hover:text-purple-700">
                                                    <flux:icon name="arrow-uturn-left" class="w-4 h-4" />
                                                </flux:button>
                                            </form>
                                        @endif
                                            <form method="POST" action="{{ route('admin.transactions.destroy', $transaction) }}" class="inline" onsubmit="return confirm('Are you sure you want to delete this transaction? This action cannot be undone.')">
                                                @csrf
                                                @method('DELETE')
                                                <flux:button type="submit" variant="outline" size="sm" class="text-red-600 hover:text-red-700">
                                                    <flux:icon name="trash" class="w-4 h-4" />
                                                </flux:button>
                                            </form>
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="7" class="px-6 py-12 text-center">
                                    <div class="text-gray-500">
                                        <flux:icon name="inbox" class="w-12 h-12 mx-auto mb-4 text-gray-300" />
                                        <p class="text-lg font-medium">No transactions found</p>
                                        <p class="text-sm">Try adjusting your search criteria or create a new transaction.</p>
                                    </div>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            @if($transactions->hasPages())
                <div class="px-6 py-4 border-t border-gray-200">
                    {{ $transactions->links() }}
                </div>
            @endif
        </div>
    </div>
@endsection
