@extends('components.layouts.admin')

@section('title', 'Transaction Generator')

@section('content')
<div class="space-y-6">
    <!-- Success Message -->
    @if (session('success'))
        <x-admin.message type="success">
            {{ session('success') }}
        </x-admin.message>
    @endif

    <!-- Error Message -->
    @if (session('error'))
        <x-admin.message type="error">
            {{ session('error') }}
        </x-admin.message>
    @endif

    <!-- Header -->
    <div class="flex justify-between items-center">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Transaction Generator</h1>
            <p class="text-gray-600">Generate sample transactions for testing and demonstration purposes</p>
        </div>
    </div>

    <!-- Generator Form -->
    <div class="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
        <form method="POST" action="{{ route('admin.transaction-generator.generate') }}" id="generatorForm">
            @csrf
            
            <!-- Account Selection -->
            <div class="mb-6">
                <h3 class="text-lg font-semibold text-gray-900 mb-4">Account Selection</h3>
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <!-- Generate for All Accounts -->
                    <div class="flex items-center">
                        <flux:checkbox 
                            id="generate_all" 
                            name="generate_all" 
                            value="1"
                            onchange="toggleAccountSelection()"
                        />
                        <flux:label for="generate_all" class="ml-2">
                            Generate for all active accounts
                        </flux:label>
                    </div>
                </div>

                <!-- Account Selection (hidden when generate_all is checked) -->
                <div id="accountSelection" class="mt-4">
                    <flux:field>
                        <flux:label>Select Accounts</flux:label>
                        <flux:select name="account_ids[]" multiple class="h-32" id="accountSelect">
                            @foreach($accounts as $account)
                                <option value="{{ $account->id }}">
                                    {{ $account->account_number }} - {{ $account->account_name }} ({{ $account->user->name }})
                                </option>
                            @endforeach
                        </flux:select>
                        <flux:error name="account_ids" />
                        <flux:description>Hold Ctrl/Cmd to select multiple accounts</flux:description>
                    </flux:field>
                </div>
            </div>

            <!-- Date Range -->
            <div class="mb-6">
                <h3 class="text-lg font-semibold text-gray-900 mb-4">Date Range</h3>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <flux:field>
                        <flux:label>Start Date</flux:label>
                        <flux:date-picker 
                            name="start_date" 
                            value="{{ old('start_date', now()->subDays(30)->format('Y-m-d')) }}"
                            required
                        />
                        <flux:error name="start_date" />
                    </flux:field>

                    <flux:field>
                        <flux:label>End Date</flux:label>
                        <flux:date-picker 
                            name="end_date" 
                            value="{{ old('end_date', now()->format('Y-m-d')) }}"
                            required
                        />
                        <flux:error name="end_date" />
                    </flux:field>
                </div>
            </div>

            <!-- Transaction Settings -->
            <div class="mb-6">
                <h3 class="text-lg font-semibold text-gray-900 mb-4">Transaction Settings</h3>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <flux:field>
                        <flux:label>Number of Transactions</flux:label>
                        <flux:input 
                            name="transaction_count" 
                            type="number"
                            min="1"
                            max="1000"
                            value="{{ old('transaction_count', 10) }}"
                            required
                        />
                        <flux:error name="transaction_count" />
                        <flux:description>Maximum 1000 transactions per generation</flux:description>
                    </flux:field>

                    <flux:field>
                        <flux:label>Transaction Direction</flux:label>
                        <flux:select name="direction" required>
                            <option value="mixed" {{ old('direction', 'mixed') === 'mixed' ? 'selected' : '' }}>Mixed (Incoming & Outgoing)</option>
                            <option value="incoming" {{ old('direction') === 'incoming' ? 'selected' : '' }}>Incoming Only</option>
                            <option value="outgoing" {{ old('direction') === 'outgoing' ? 'selected' : '' }}>Outgoing Only</option>
                        </flux:select>
                        <flux:error name="direction" />
                    </flux:field>

                    <flux:field>
                        <flux:label>Preview</flux:label>
                        <flux:button 
                            type="button" 
                            variant="outline" 
                            onclick="previewTransactions()"
                            class="w-full"
                        >
                            <flux:icon name="eye" class="w-4 h-4 mr-2" />
                            Preview Sample
                        </flux:button>
                    </flux:field>
                </div>
            </div>

            <!-- Transaction Types Info -->
            <div class="mb-6">
                <h3 class="text-lg font-semibold text-gray-900 mb-4">Transaction Types</h3>
                <div class="bg-gray-50 rounded-lg p-4">
                    <p class="text-sm text-gray-600 mb-2">The generator will create the following types of transactions:</p>
                    <div class="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                        <div>
                            <strong>Deposits:</strong> Wire, ACH, Bank Transfer, Digital Wallet, Interest, Refund
                        </div>
                        <div>
                            <strong>Withdrawals:</strong> Wire, ACH, Bank Transfer, Digital Wallet
                        </div>
                        <div>
                            <strong>Transfers:</strong> Wire, ACH, SWIFT, SEPA, Domestic
                        </div>
                        <div>
                            <strong>Payments:</strong> Bill, Merchant, Utility, Loan
                        </div>
                        <div>
                            <strong>Refunds:</strong> Merchant, Service, Fee
                        </div>
                        <div>
                            <strong>Fees:</strong> Maintenance, Transfer, Overdraft, Late
                        </div>
                        <div>
                            <strong>Interest:</strong> Savings, Deposit, Loan
                        </div>
                        <div class="text-gray-500">
                            <em>Excludes: Card payments, ATM, Inter-account, Cheque deposits</em>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Form Actions -->
            <div class="flex justify-end space-x-3">
                <flux:button type="button" variant="outline" onclick="resetForm()">
                    Reset
                </flux:button>
                <flux:button icon="plus" type="submit" variant="primary" onclick="return confirmGeneration()">
                    Generate Transactions
                </flux:button>
            </div>
        </form>
    </div>

    <!-- Preview Modal -->
    <div id="previewModal" class="fixed inset-0 bg-gray-900/50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg shadow-xl max-w-4xl w-full max-h-[80vh] overflow-hidden">
                <div class="flex justify-between items-center p-6 border-b border-gray-200">
                    <h3 class="text-lg font-semibold text-gray-900">Transaction Preview</h3>
                    <button onclick="closePreview()" class="text-gray-400 hover:text-gray-600">
                        <flux:icon name="x-mark" class="w-6 h-6" />
                    </button>
                </div>
                <div class="p-6 overflow-y-auto max-h-[60vh]">
                    <div id="previewContent">
                        <div class="flex items-center justify-center py-8">
                            <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                            <span class="ml-2 text-gray-600">Loading preview...</span>
                        </div>
                    </div>
                </div>
                <div class="flex justify-end p-6 border-t border-gray-200">
                    <flux:button variant="outline" onclick="closePreview()">
                        Close
                    </flux:button>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function toggleAccountSelection() {
    const generateAll = document.getElementById('generate_all').checked;
    const accountSelection = document.getElementById('accountSelection');
    
    if (generateAll) {
        accountSelection.style.display = 'none';
        document.getElementById('accountSelect').required = false;
    } else {
        accountSelection.style.display = 'block';
        document.getElementById('accountSelect').required = true;
    }
}

function previewTransactions() {
    const form = document.getElementById('generatorForm');
    const formData = new FormData(form);
    
    // Get the first selected account for preview
    const accountSelect = document.getElementById('accountSelect');
    const selectedAccounts = Array.from(accountSelect.selectedOptions);
    
    if (selectedAccounts.length === 0 && !document.getElementById('generate_all').checked) {
        alert('Please select at least one account for preview');
        return;
    }
    
    const accountId = selectedAccounts.length > 0 ? selectedAccounts[0].value : 1;
    
    // Show modal
    document.getElementById('previewModal').classList.remove('hidden');
    
    // Make AJAX request
    fetch('{{ route("admin.transaction-generator.preview") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            account_id: accountId,
            start_date: formData.get('start_date'),
            end_date: formData.get('end_date'),
            transaction_count: formData.get('transaction_count'),
            direction: formData.get('direction')
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            displayPreview(data.preview);
        } else {
            document.getElementById('previewContent').innerHTML = 
                '<div class="text-red-600 text-center py-4">Error: ' + data.message + '</div>';
        }
    })
    .catch(error => {
        document.getElementById('previewContent').innerHTML = 
            '<div class="text-red-600 text-center py-4">Error loading preview</div>';
    });
}

function displayPreview(transactions) {
    const content = document.getElementById('previewContent');
    
    let html = '<div class="space-y-4">';
    html += '<div class="text-sm text-gray-600 mb-4">Showing sample of ' + transactions.length + ' transactions:</div>';
    
    transactions.forEach((tx, index) => {
        const amountClass = tx.direction === 'incoming' ? 'text-green-600' : 'text-red-600';
        const amountPrefix = tx.direction === 'incoming' ? '+' : '-';
        
        html += `
            <div class="border border-gray-200 rounded-lg p-4">
                <div class="flex justify-between items-start">
                    <div>
                        <div class="font-medium text-gray-900">${tx.description}</div>
                        <div class="text-sm text-gray-500">${tx.type} • ${tx.subcategory}</div>
                        <div class="text-xs text-gray-400">${tx.date}</div>
                    </div>
                    <div class="text-right">
                        <div class="font-semibold ${amountClass}">${amountPrefix}$${tx.amount.toFixed(2)} ${tx.currency}</div>
                        <div class="text-xs text-gray-500">
                            Fee: $${tx.fee_amount.toFixed(2)} | Tax: $${tx.tax_amount.toFixed(2)}
                        </div>
                        <div class="text-xs text-gray-500">Net: $${tx.net_amount.toFixed(2)}</div>
                    </div>
                </div>
            </div>
        `;
    });
    
    html += '</div>';
    content.innerHTML = html;
}

function closePreview() {
    document.getElementById('previewModal').classList.add('hidden');
}

function resetForm() {
    document.getElementById('generatorForm').reset();
    toggleAccountSelection();
}

function confirmGeneration() {
    const count = document.querySelector('input[name="transaction_count"]').value;
    const generateAll = document.getElementById('generate_all').checked;
    
    let message = `Are you sure you want to generate ${count} transactions?`;
    
    if (generateAll) {
        message += '\n\nThis will generate transactions for ALL active accounts.';
    }
    
    return confirm(message);
}

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    toggleAccountSelection();
});
</script>
@endsection
