@extends('components.layouts.admin')

@section('title', 'Transaction Analytics')

@section('content')
<div class="space-y-6">
    <!-- Success Message -->
    @if (session('success'))
        <x-admin.message type="success">
            {{ session('success') }}
        </x-admin.message>
    @endif

    <!-- Error Message -->
    @if (session('error'))
        <x-admin.message type="error">
            {{ session('error') }}
        </x-admin.message>
    @endif

    <!-- Header -->
    <div class="flex justify-between items-center">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Transaction Analytics</h1>
            <p class="text-gray-600">Transaction volume, trends, and performance insights</p>
        </div>
        <div class="flex space-x-3">
            <flux:select id="period-select" onchange="updatePeriod(this.value)">
                <option value="7d" {{ $period === '7d' ? 'selected' : '' }}>Last 7 Days</option>
                <option value="30d" {{ $period === '30d' ? 'selected' : '' }}>Last 30 Days</option>
                <option value="90d" {{ $period === '90d' ? 'selected' : '' }}>Last 90 Days</option>
                <option value="1y" {{ $period === '1y' ? 'selected' : '' }}>Last Year</option>
            </flux:select>
            <flux:button href="{{ route('admin.analytics.index') }}" variant="outline" class="text-xs">
                <flux:icon name="arrow-left" class="w-4 h-4 mr-2" />
                Back to Analytics
            </flux:button>
        </div>
    </div>

    <!-- Transaction Summary Cards -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <flux:card>
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <flux:icon name="arrow-path" class="w-8 h-8 text-blue-600" />
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Total Transactions</p>
                    <p class="text-2xl font-semibold text-gray-900">{{ number_format($transactionTrends['total_transactions']) }}</p>
                </div>
            </div>
        </flux:card>

        <flux:card>
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <flux:icon name="chart-line" class="w-8 h-8 text-green-600" />
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Period Transactions</p>
                    <p class="text-2xl font-semibold text-gray-900">{{ number_format($transactionTrends['period_transactions']) }}</p>
                </div>
            </div>
        </flux:card>

        <flux:card>
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <flux:icon name="currency-dollar" class="w-8 h-8 text-purple-600" />
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Total Volume</p>
                    <p class="text-2xl font-semibold text-gray-900">${{ number_format($transactionTrends['total_volume'], 2) }}</p>
                </div>
            </div>
        </flux:card>

        <flux:card>
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <flux:icon name="calculator" class="w-8 h-8 text-orange-600" />
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Average Transaction</p>
                    <p class="text-2xl font-semibold text-gray-900">${{ number_format($transactionTrends['average_transaction'], 2) }}</p>
                </div>
            </div>
        </flux:card>
    </div>

    <!-- Growth Rate Indicators -->
    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
        <flux:card>
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-gray-500">Transaction Growth Rate</p>
                    <p class="text-3xl font-bold {{ $transactionTrends['transaction_growth_rate'] >= 0 ? 'text-green-600' : 'text-red-600' }}">
                        {{ $transactionTrends['transaction_growth_rate'] >= 0 ? '+' : '' }}{{ $transactionTrends['transaction_growth_rate'] }}%
                    </p>
                </div>
                <flux:icon name="chart-line" class="w-12 h-12 text-gray-400" />
            </div>
        </flux:card>

        <flux:card>
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-gray-500">Volume Growth Rate</p>
                    <p class="text-3xl font-bold {{ $transactionTrends['volume_growth_rate'] >= 0 ? 'text-green-600' : 'text-red-600' }}">
                        {{ $transactionTrends['volume_growth_rate'] >= 0 ? '+' : '' }}{{ $transactionTrends['volume_growth_rate'] }}%
                    </p>
                </div>
                <flux:icon name="currency-dollar" class="w-12 h-12 text-gray-400" />
            </div>
        </flux:card>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <!-- Transaction Volume Chart -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Transaction Volume Trend</flux:heading>
            @if($transactionVolume->count() > 0)
                <div class="space-y-3">
                    @foreach($transactionVolume as $data)
                        <div class="flex justify-between items-center">
                            <span class="text-sm font-medium text-gray-900">
                                {{ \Carbon\Carbon::parse($data->date)->format('M j, Y') }}
                            </span>
                            <div class="flex items-center space-x-2">
                                <div class="w-32 bg-gray-200 rounded-full h-2">
                                    <div class="bg-blue-600 h-2 rounded-full" style="width: {{ ($data->count / $transactionVolume->max('count')) * 100 }}%"></div>
                                </div>
                                <span class="text-sm text-gray-500 w-16 text-right">{{ $data->count }}</span>
                            </div>
                        </div>
                    @endforeach
                </div>
            @else
                <p class="text-sm text-gray-500">No transaction volume data available for the selected period.</p>
            @endif
        </flux:card>

        <!-- Transaction Volume Amount Chart -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Transaction Amount Trend</flux:heading>
            @if($transactionVolume->count() > 0)
                <div class="space-y-3">
                    @foreach($transactionVolume as $data)
                        <div class="flex justify-between items-center">
                            <span class="text-sm font-medium text-gray-900">
                                {{ \Carbon\Carbon::parse($data->date)->format('M j, Y') }}
                            </span>
                            <div class="flex items-center space-x-2">
                                <div class="w-32 bg-gray-200 rounded-full h-2">
                                    <div class="bg-green-600 h-2 rounded-full" style="width: {{ ($data->total_amount / $transactionVolume->max('total_amount')) * 100 }}%"></div>
                                </div>
                                <span class="text-sm text-gray-500 w-20 text-right">${{ number_format($data->total_amount, 0) }}</span>
                            </div>
                        </div>
                    @endforeach
                </div>
            @else
                <p class="text-sm text-gray-500">No transaction amount data available for the selected period.</p>
            @endif
        </flux:card>
    </div>

    <!-- Transaction Type Analysis -->
    <flux:card>
        <flux:heading size="lg" class="mb-4">Transaction Type Analysis</flux:heading>
        @if($transactionTrends['by_type']->count() > 0)
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                @foreach($transactionTrends['by_type'] as $type)
                    <div class="p-4 bg-gray-50 rounded-lg">
                        <div class="flex justify-between items-center mb-2">
                            <span class="text-sm font-medium text-gray-900">
                                {{ ucfirst(str_replace('_', ' ', $type->type)) }}
                            </span>
                            <span class="text-xs text-gray-500">{{ $type->count }} transactions</span>
                        </div>
                        <div class="text-2xl font-bold text-gray-900 mb-1">
                            ${{ number_format($type->total_amount, 2) }}
                        </div>
                        <div class="text-sm text-gray-500">
                            {{ $transactionTrends['period_transactions'] > 0 ? number_format(($type->count / $transactionTrends['period_transactions']) * 100, 1) : 0 }}% of total
                        </div>
                    </div>
                @endforeach
            </div>
        @else
            <p class="text-sm text-gray-500">No transaction type data available for the selected period.</p>
        @endif
    </flux:card>

    <!-- Transaction Status Analysis -->
    <flux:card>
        <flux:heading size="lg" class="mb-4">Transaction Status Analysis</flux:heading>
        @if($transactionTrends['by_status']->count() > 0)
            <div class="space-y-4">
                @foreach($transactionTrends['by_status'] as $status)
                    <div class="flex justify-between items-center">
                        <span class="text-sm font-medium text-gray-900">
                            {{ ucfirst(str_replace('_', ' ', $status->status)) }}
                        </span>
                        <div class="flex items-center space-x-2">
                            <div class="w-48 bg-gray-200 rounded-full h-2">
                                <div class="h-2 rounded-full {{ 
                                    $status->status === 'completed' ? 'bg-green-600' : 
                                    ($status->status === 'pending' ? 'bg-yellow-600' : 
                                    ($status->status === 'failed' ? 'bg-red-600' : 
                                    ($status->status === 'cancelled' ? 'bg-gray-600' : 'bg-blue-600'))) 
                                }}" style="width: {{ ($status->count / $transactionTrends['by_status']->sum('count')) * 100 }}%"></div>
                            </div>
                            <span class="text-sm text-gray-500 w-16 text-right">{{ $status->count }}</span>
                            <span class="text-sm text-gray-500 w-20 text-right">${{ number_format($status->total_amount, 0) }}</span>
                        </div>
                    </div>
                @endforeach
            </div>
        @else
            <p class="text-sm text-gray-500">No transaction status data available for the selected period.</p>
        @endif
    </flux:card>

    <!-- Performance Metrics -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <!-- Volume Metrics -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Volume Metrics</flux:heading>
            <div class="space-y-4">
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Period Volume</span>
                    <span class="text-lg font-semibold text-gray-900">${{ number_format($transactionTrends['period_volume'], 2) }}</span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Average per Transaction</span>
                    <span class="text-lg font-semibold text-gray-900">${{ number_format($transactionTrends['average_transaction'], 2) }}</span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Daily Average</span>
                    <span class="text-lg font-semibold text-gray-900">
                        ${{ number_format($transactionTrends['period_volume'] / max(1, $transactionVolume->count()), 2) }}
                    </span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Peak Day Volume</span>
                    <span class="text-lg font-semibold text-gray-900">
                        ${{ number_format($transactionVolume->max('total_amount'), 2) }}
                    </span>
                </div>
            </div>
        </flux:card>

        <!-- Activity Metrics -->
        <flux:card>
            <flux:heading size="lg" class="mb-4">Activity Metrics</flux:heading>
            <div class="space-y-4">
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Period Transactions</span>
                    <span class="text-lg font-semibold text-gray-900">{{ number_format($transactionTrends['period_transactions']) }}</span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Daily Average</span>
                    <span class="text-lg font-semibold text-gray-900">
                        {{ number_format($transactionTrends['period_transactions'] / max(1, $transactionVolume->count()), 1) }}
                    </span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Peak Day Count</span>
                    <span class="text-lg font-semibold text-gray-900">{{ $transactionVolume->max('count') }}</span>
                </div>
                <div class="flex justify-between items-center">
                    <span class="text-sm text-gray-700">Success Rate</span>
                    <span class="text-lg font-semibold text-gray-900">
                        @php
                            $completed = $transactionTrends['by_status']->where('status', 'completed')->first();
                            $total = $transactionTrends['by_status']->sum('count');
                            $successRate = $completed && $total > 0 ? ($completed->count / $total) * 100 : 0;
                        @endphp
                        {{ number_format($successRate, 1) }}%
                    </span>
                </div>
            </div>
        </flux:card>
    </div>
</div>

<script>
function updatePeriod(period) {
    const url = new URL(window.location);
    url.searchParams.set('period', period);
    window.location.href = url.toString();
}
</script>
@endsection
