<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('beneficiaries', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            
            // Beneficiary Information
            $table->string('beneficiary_id')->unique(); // External beneficiary ID
            $table->string('name');
            $table->string('type'); // individual, business, bank, international
            
            // Contact Information
            $table->string('email')->nullable();
            $table->string('phone')->nullable();
            $table->text('address')->nullable();
            $table->string('city')->nullable();
            $table->string('state_province')->nullable();
            $table->string('postal_code')->nullable();
            $table->string('country')->nullable();
            
            // Banking Information
            $table->string('bank_name')->nullable();
            $table->string('bank_code')->nullable(); // SWIFT, BIC, routing number
            $table->string('account_number')->nullable();
            $table->string('iban')->nullable(); // International Bank Account Number
            $table->string('currency', 3)->default('USD');
            
            // Business Specific Information
            $table->string('business_name')->nullable();
            $table->string('business_type')->nullable();
            $table->string('business_registration_number')->nullable();
            $table->string('tax_identification_number')->nullable();
            $table->string('website')->nullable();
            
            // Verification and Compliance
            $table->enum('verification_status', [
                'pending', 'verified', 'rejected', 'under_review', 'expired'
            ])->default('pending');
            $table->enum('verification_level', ['basic', 'enhanced', 'full'])->default('basic');
            $table->timestamp('verified_at')->nullable();
            $table->foreignId('verified_by')->nullable()->constrained('users');
            $table->text('verification_notes')->nullable();
            
            // Risk Assessment
            $table->enum('risk_level', ['low', 'medium', 'high'])->default('medium');
            $table->boolean('aml_verified')->default(false);
            $table->boolean('sanctions_verified')->default(false);
            $table->timestamp('last_risk_assessment')->nullable();
            $table->text('risk_notes')->nullable();
            
            // Transfer Limits and Preferences
            $table->decimal('daily_transfer_limit', 20, 4)->nullable();
            $table->decimal('monthly_transfer_limit', 20, 4)->nullable();
            $table->decimal('single_transfer_limit', 20, 4)->nullable();
            $table->enum('transfer_methods', ['wire', 'ach', 'swift', 'sepa', 'instant'])->nullable();
            $table->string('preferred_transfer_method')->nullable();
            
            // Account Association
            $table->json('allowed_accounts')->nullable(); // Which user accounts can transfer to this beneficiary
            $table->boolean('active')->default(true);
            $table->timestamp('last_transfer_at')->nullable();
            $table->integer('total_transfers')->default(0);
            $table->decimal('total_transfer_amount', 20, 4)->default(0);
            
            // Security and Preferences
            $table->boolean('two_factor_required')->default(true);
            $table->boolean('notification_enabled')->default(true);
            $table->json('notification_preferences')->nullable();
            $table->json('security_settings')->nullable();
            
            // Additional Information
            $table->text('notes')->nullable();
            $table->json('metadata')->nullable(); // Additional beneficiary data
            $table->string('relationship')->nullable(); // family, friend, business_partner, etc.
            
            // Expiration and Maintenance
            $table->timestamp('expires_at')->nullable();
            $table->boolean('auto_renewal')->default(false);
            $table->timestamp('last_updated_at')->nullable();
            
            // Timestamps
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index(['user_id']);
            $table->index(['beneficiary_id']);
            $table->index(['type']);
            $table->index(['verification_status']);
            $table->index(['risk_level']);
            $table->index(['country']);
            $table->index(['currency']);
            $table->index(['active']);
            $table->index(['expires_at']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('beneficiaries');
    }
};
