<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('transactions', function (Blueprint $table) {
            $table->id();
            
            // Transaction Identification
            $table->string('transaction_id')->unique(); // External transaction ID
            $table->string('reference_number')->nullable(); // Internal reference
            $table->string('external_reference')->nullable(); // External system reference
            
            // Account Information
            $table->foreignId('from_account_id')->nullable()->constrained('accounts');
            $table->foreignId('to_account_id')->nullable()->constrained('accounts');
            $table->foreignId('user_id')->constrained(); // Primary user involved
            
            // Transaction Details
            $table->enum('type', [
                'deposit', 'withdrawal', 'transfer', 'payment', 'refund', 
                'fee', 'interest', 'adjustment', 'exchange', 'investment',
                'loan_disbursement', 'loan_repayment', 'card_payment', 'atm'
            ]);
            $table->enum('category', [
                'banking', 'investment', 'loan', 'card', 'atm', 'exchange',
                'fee', 'interest', 'adjustment', 'other'
            ])->default('banking');
            $table->string('subcategory')->nullable(); // More specific categorization
            
            // Amount and Currency
            $table->decimal('amount', 20, 4); // Transaction amount
            $table->string('currency', 3); // Transaction currency
            $table->decimal('exchange_rate', 15, 6)->nullable(); // If currency conversion involved
            $table->decimal('converted_amount', 20, 4)->nullable(); // Amount in account currency
            $table->decimal('fee_amount', 10, 4)->default(0);
            $table->decimal('tax_amount', 10, 4)->default(0);
            $table->decimal('net_amount', 20, 4)->nullable(); // Amount after fees and taxes
            
            // Transaction Status
            $table->enum('status', [
                'pending', 'processing', 'completed', 'failed', 'cancelled',
                'reversed', 'disputed', 'under_review'
            ])->default('pending');
            $table->enum('verification_status', ['pending', 'verified', 'rejected'])->default('pending');
            $table->timestamp('processed_at')->nullable();
            $table->timestamp('completed_at')->nullable();
            
            // Transaction Details
            $table->text('description')->nullable();
            $table->text('notes')->nullable();
            $table->json('metadata')->nullable(); // Additional transaction data
            
            // Security and Verification
            $table->string('ip_address')->nullable();
            $table->string('user_agent')->nullable();
            $table->string('device_id')->nullable();
            $table->string('location')->nullable();
            $table->boolean('two_factor_verified')->default(false);
            $table->boolean('biometric_verified')->default(false);
            
            // Risk and Compliance
            $table->enum('risk_level', ['low', 'medium', 'high'])->default('low');
            $table->boolean('aml_flagged')->default(false);
            $table->text('aml_notes')->nullable();
            $table->boolean('fraud_flagged')->default(false);
            $table->text('fraud_notes')->nullable();
            
            // Transfer Specific Fields
            $table->string('transfer_method')->nullable(); // wire, ach, swift, sepa, etc.
            $table->string('transfer_speed')->nullable(); // instant, same_day, next_day, standard
            $table->timestamp('estimated_arrival')->nullable();
            $table->timestamp('actual_arrival')->nullable();
            
            // Payment Specific Fields
            $table->string('payment_method')->nullable(); // card, bank_transfer, digital_wallet
            $table->string('merchant_name')->nullable();
            $table->string('merchant_id')->nullable();
            $table->string('card_last_four')->nullable();
            $table->string('card_type')->nullable();
            
            // Investment Specific Fields
            $table->string('investment_type')->nullable();
            $table->string('investment_symbol')->nullable();
            $table->decimal('units', 15, 6)->nullable();
            $table->decimal('unit_price', 15, 4)->nullable();
            
            // Reversal and Dispute Information
            $table->foreignId('reversed_transaction_id')->nullable()->constrained('transactions');
            $table->text('reversal_reason')->nullable();
            $table->timestamp('reversed_at')->nullable();
            $table->foreignId('disputed_transaction_id')->nullable()->constrained('transactions');
            $table->text('dispute_reason')->nullable();
            $table->enum('dispute_status', ['open', 'under_review', 'resolved', 'closed'])->nullable();
            
            // Batch Processing
            $table->string('batch_id')->nullable();
            $table->integer('batch_sequence')->nullable();
            
            // Timestamps
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index(['transaction_id']);
            $table->index(['user_id']);
            $table->index(['from_account_id']);
            $table->index(['to_account_id']);
            $table->index(['type']);
            $table->index(['category']);
            $table->index(['status']);
            $table->index(['currency']);
            $table->index(['created_at']);
            $table->index(['batch_id']);
            $table->index(['reference_number']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('transactions');
    }
};
