<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('accounts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            
            // Account Information
            $table->string('account_number')->unique();
            $table->string('account_type'); // savings, checking, fixed_deposit, business, investment
            $table->string('currency', 3); // USD, EUR, GBP, etc.
            $table->string('account_name')->nullable();
            $table->text('description')->nullable();
            
            // Balance and Limits
            $table->decimal('balance', 20, 4)->default(0); // Using 4 decimal places for precision
            $table->decimal('available_balance', 20, 4)->default(0);
            $table->decimal('hold_balance', 20, 4)->default(0);
            $table->decimal('daily_transfer_limit', 20, 4)->nullable();
            $table->decimal('monthly_transfer_limit', 20, 4)->nullable();
            $table->decimal('daily_withdrawal_limit', 20, 4)->nullable();
            $table->decimal('monthly_withdrawal_limit', 20, 4)->nullable();
            
            // Account Status
            $table->enum('status', ['active', 'inactive', 'suspended', 'closed', 'pending'])->default('pending');
            $table->enum('verification_status', ['pending', 'verified', 'rejected'])->default('pending');
            $table->timestamp('activated_at')->nullable();
            $table->timestamp('closed_at')->nullable();
            $table->text('closure_reason')->nullable();
            
            // Interest and Fees
            $table->decimal('interest_rate', 5, 4)->nullable(); // 5.2500% = 0.0525
            $table->enum('interest_frequency', ['monthly', 'quarterly', 'annually'])->nullable();
            $table->decimal('monthly_maintenance_fee', 10, 2)->nullable();
            $table->decimal('overdraft_limit', 20, 4)->nullable();
            $table->decimal('overdraft_fee', 10, 2)->nullable();
            
            // Account Features
            $table->boolean('online_banking_enabled')->default(true);
            $table->boolean('mobile_banking_enabled')->default(true);
            $table->boolean('debit_card_enabled')->default(false);
            $table->boolean('credit_card_enabled')->default(false);
            $table->boolean('international_transfers_enabled')->default(false);
            $table->boolean('investment_trading_enabled')->default(false);
            
            // Security Settings
            $table->boolean('two_factor_required')->default(true);
            $table->boolean('biometric_required')->default(false);
            $table->json('security_preferences')->nullable();
            
            // Compliance and Risk
            $table->enum('risk_level', ['low', 'medium', 'high'])->default('medium');
            $table->boolean('aml_monitoring_enabled')->default(true);
            $table->timestamp('last_risk_assessment')->nullable();
            
            // Account Hierarchy (for business accounts)
            $table->foreignId('parent_account_id')->nullable()->constrained('accounts');
            $table->boolean('is_master_account')->default(false);
            
            // Business Account Specific
            $table->string('business_name')->nullable();
            $table->string('business_type')->nullable();
            $table->string('business_registration_number')->nullable();
            $table->string('tax_identification_number')->nullable();
            
            // Investment Account Specific
            $table->string('investment_strategy')->nullable();
            $table->enum('risk_tolerance', ['conservative', 'moderate', 'aggressive'])->nullable();
            $table->json('investment_preferences')->nullable();
            
            // Timestamps
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index(['user_id']);
            $table->index(['account_number']);
            $table->index(['account_type']);
            $table->index(['currency']);
            $table->index(['status']);
            $table->index(['verification_status']);
            $table->index(['parent_account_id']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('accounts');
    }
};
