<?php

namespace App\Services;

use App\Models\Transaction;
use App\Models\Account;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Akaunting\Money\Money;
use Akaunting\Money\Currency;

class TransactionService
{
    /**
     * Create a new transaction with proper validation and account updates.
     */
    public function createTransaction(array $data): Transaction
    {
        return DB::transaction(function () use ($data) {
            // Generate transaction IDs
            $transactionId = 'TXN-' . now()->format('YmdHis') . '-' . strtoupper(Str::random(6));
            $referenceNumber = 'REF-' . now()->format('YmdHis') . '-' . strtoupper(Str::random(4));

            // Create the transaction
            $transaction = Transaction::create([
                'transaction_id' => $transactionId,
                'reference_number' => $referenceNumber,
                'external_reference' => $data['external_reference'] ?? null,
                'user_id' => $data['user_id'],
                'from_account_id' => $data['from_account_id'] ?? null,
                'to_account_id' => $data['to_account_id'] ?? null,
                'type' => $data['type'],
                'category' => $data['category'],
                'subcategory' => $data['subcategory'] ?? null,
                'amount' => $data['amount'],
                'currency' => $data['currency'],
                'exchange_rate' => $data['exchange_rate'] ?? null,
                'converted_amount' => $data['converted_amount'] ?? null,
                'fee_amount' => $data['fee_amount'] ?? 0,
                'tax_amount' => $data['tax_amount'] ?? 0,
                'net_amount' => $data['net_amount'] ?? $data['amount'],
                'description' => $data['description'],  
                'notes' => $data['notes'] ?? null,
                'status' => $data['status'] ?? 'pending',
                'transfer_method' => $data['transfer_method'] ?? null,
                'transfer_speed' => $data['transfer_speed'] ?? null,
                'estimated_arrival' => $data['estimated_arrival'] ?? null,
                'actual_arrival' => $data['actual_arrival'] ?? null,
                'external_reference' => $data['external_reference'] ?? null,
                'verification_status' => $data['verification_status'] ?? 'pending',
                'ip_address' => $data['ip_address'] ?? null,
                'processed_at' => ($data['status'] ?? 'pending') === 'completed' ? now() : null,
                'completed_at' => ($data['status'] ?? 'pending') === 'completed' ? now() : null,
            ]);

            // Update account balances if transaction is completed
            if ($transaction->status === 'completed') {
                $this->updateAccountBalances($transaction);
            }

            // Log the transaction creation
            Log::info('Transaction created', [
                'transaction_id' => $transaction->transaction_id,
                'type' => $transaction->type,
                'amount' => $transaction->amount,
                'currency' => $transaction->currency,
                'user_id' => $transaction->user_id,
            ]);

            return $transaction;
        });
    }

    /**
     * Update an existing transaction.
     */
    public function updateTransaction(Transaction $transaction, array $data): Transaction
    {
        return DB::transaction(function () use ($transaction, $data) {
            $oldStatus = $transaction->status;
            $oldAmount = $transaction->amount;

            // Update the transaction
            $transaction->update([
                'external_reference' => $data['external_reference'] ?? $transaction->external_reference,
                'user_id' => $data['user_id'],
                'from_account_id' => $data['from_account_id'] ?? $transaction->from_account_id,
                'to_account_id' => $data['to_account_id'] ?? $transaction->to_account_id,
                'type' => $data['type'],
                'category' => $data['category'],
                'subcategory' => $data['subcategory'] ?? $transaction->subcategory,
                'amount' => $data['amount'],
                'currency' => $data['currency'],
                'exchange_rate' => $data['exchange_rate'] ?? $transaction->exchange_rate,
                'converted_amount' => $data['converted_amount'] ?? $transaction->converted_amount,
                'fee_amount' => $data['fee_amount'] ?? $transaction->fee_amount,
                'tax_amount' => $data['tax_amount'] ?? $transaction->tax_amount,
                'net_amount' => $data['net_amount'] ?? $data['amount'],
                'description' => $data['description'],
                'notes' => $data['notes'] ?? $transaction->notes,
                'status' => $data['status'],
                'transfer_method' => $data['transfer_method'] ?? $transaction->transfer_method,
                'transfer_speed' => $data['transfer_speed'] ?? $transaction->transfer_speed,
                'estimated_arrival' => $data['estimated_arrival'] ?? $transaction->estimated_arrival,
                'actual_arrival' => $data['actual_arrival'] ?? $transaction->actual_arrival,
                'external_reference' => $data['external_reference'] ?? $transaction->external_reference,
                'verification_status' => $data['verification_status'],
                'ip_address' => $data['ip_address'] ?? $transaction->ip_address,
                'created_at' => isset($data['created_at']) && $data['created_at'] ? \Carbon\Carbon::createFromFormat('Y-m-d\TH:i', $data['created_at']) : $transaction->created_at,
                'processed_at' => $data['status'] === 'completed' && $oldStatus !== 'completed' ? now() : $transaction->processed_at,
                'completed_at' => $data['status'] === 'completed' && $oldStatus !== 'completed' ? now() : $transaction->completed_at,
            ]);

            // Handle status changes
            if ($oldStatus !== $data['status']) {
                $this->handleStatusChange($transaction, $oldStatus, $data['status']);
            }

            // Handle amount changes
            if ($oldAmount !== $data['amount'] && $transaction->status === 'completed') {
                $this->recalculateAccountBalances($transaction, $oldAmount, $data['amount']);
            }

            // Log the transaction update
            Log::info('Transaction updated', [
                'transaction_id' => $transaction->transaction_id,
                'old_status' => $oldStatus,
                'new_status' => $data['status'],
                'old_amount' => $oldAmount,
                'new_amount' => $data['amount'],
            ]);

            return $transaction;
        });
    }

    /**
     * Approve a pending transaction.
     */
    public function approveTransaction(Transaction $transaction): Transaction
    {
        if ($transaction->status !== 'pending') {
            throw new \Exception('Only pending transactions can be approved.');
        }

        return DB::transaction(function () use ($transaction) {
            $transaction->update([
                'status' => 'completed',
                'verification_status' => 'verified',
                'processed_at' => now(),
                'completed_at' => now(),
            ]);

            // Update account balances
            $this->updateAccountBalances($transaction);

            // Log the approval
            Log::info('Transaction approved', [
                'transaction_id' => $transaction->transaction_id,
                'amount' => $transaction->amount,
                'currency' => $transaction->currency,
            ]);

            return $transaction;
        });
    }

    /**
     * Reject a pending transaction.
     */
    public function rejectTransaction(Transaction $transaction, string $reason = null): Transaction
    {
        if ($transaction->status !== 'pending') {
            throw new \Exception('Only pending transactions can be rejected.');
        }

        return DB::transaction(function () use ($transaction, $reason) {
            $transaction->update([
                'status' => 'failed',
                'verification_status' => 'rejected',
                'notes' => $reason ? ($transaction->notes . "\nRejection reason: " . $reason) : $transaction->notes,
            ]);

            // Log the rejection
            Log::info('Transaction rejected', [
                'transaction_id' => $transaction->transaction_id,
                'reason' => $reason,
            ]);

            return $transaction;
        });
    }

    /**
     * Reverse a completed transaction.
     */
    public function reverseTransaction(Transaction $transaction, string $reason = null): Transaction
    {
        if ($transaction->status !== 'completed') {
            throw new \Exception('Only completed transactions can be reversed.');
        }

        return DB::transaction(function () use ($transaction, $reason) {
            $transaction->update([
                'status' => 'reversed',
                'reversed_at' => now(),
                'reversal_reason' => $reason ?? 'Admin reversal',
            ]);

            // Reverse account balance changes
            $this->reverseAccountBalances($transaction);

            // Log the reversal
            Log::info('Transaction reversed', [
                'transaction_id' => $transaction->transaction_id,
                'amount' => $transaction->amount,
                'currency' => $transaction->currency,
                'reason' => $reason,
            ]);

            return $transaction;
        });
    }

    /**
     * Update account balances based on transaction.
     */
    protected function updateAccountBalances(Transaction $transaction): void
    {
        $amount = $transaction->net_amount ?? $transaction->amount;

        switch ($transaction->type) {
            case 'deposit':
                if ($transaction->to_account_id) {
                    $this->creditAccount($transaction->toAccount, $amount);
                }
                break;

            case 'withdrawal':
                if ($transaction->from_account_id) {
                    $this->debitAccount($transaction->fromAccount, $amount);
                }
                break;

            case 'transfer':
                if ($transaction->from_account_id && $transaction->to_account_id) {
                    $this->debitAccount($transaction->fromAccount, $amount);
                    $this->creditAccount($transaction->toAccount, $amount);
                }
                break;

            case 'payment':
                if ($transaction->from_account_id) {
                    $this->debitAccount($transaction->fromAccount, $amount);
                }
                break;

            case 'fee':
                if ($transaction->from_account_id) {
                    $this->debitAccount($transaction->fromAccount, $amount);
                }
                break;

            case 'interest':
                if ($transaction->to_account_id) {
                    $this->creditAccount($transaction->toAccount, $amount);
                }
                break;

            case 'refund':
                if ($transaction->to_account_id) {
                    $this->creditAccount($transaction->toAccount, $amount);
                }
                break;
        }
    }

    /**
     * Reverse account balance changes.
     */
    protected function reverseAccountBalances(Transaction $transaction): void
    {
        $amount = $transaction->net_amount ?? $transaction->amount;

        switch ($transaction->type) {
            case 'deposit':
                if ($transaction->to_account_id) {
                    $this->debitAccount($transaction->toAccount, $amount);
                }
                break;

            case 'withdrawal':
                if ($transaction->from_account_id) {
                    $this->creditAccount($transaction->fromAccount, $amount);
                }
                break;

            case 'transfer':
                if ($transaction->from_account_id && $transaction->to_account_id) {
                    $this->creditAccount($transaction->fromAccount, $amount);
                    $this->debitAccount($transaction->toAccount, $amount);
                }
                break;

            case 'payment':
                if ($transaction->from_account_id) {
                    $this->creditAccount($transaction->fromAccount, $amount);
                }
                break;

            case 'fee':
                if ($transaction->from_account_id) {
                    $this->creditAccount($transaction->fromAccount, $amount);
                }
                break;

            case 'interest':
                if ($transaction->to_account_id) {
                    $this->debitAccount($transaction->toAccount, $amount);
                }
                break;

            case 'refund':
                if ($transaction->to_account_id) {
                    $this->debitAccount($transaction->toAccount, $amount);
                }
                break;
        }
    }

    /**
     * Credit an account.
     */
    protected function creditAccount(Account $account, float $amount): void
    {
        $account->increment('balance', $amount);
        $account->increment('available_balance', $amount);

        Log::info('Account credited', [
            'account_id' => $account->id,
            'account_number' => $account->account_number,
            'amount' => $amount,
            'new_balance' => $account->fresh()->balance,
        ]);
    }

    /**
     * Debit an account.
     */
    protected function debitAccount(Account $account, float $amount): void
    {
        if ($account->available_balance < $amount) {
            throw new \Exception('Insufficient funds in account ' . $account->account_number);
        }

        $account->decrement('balance', $amount);
        $account->decrement('available_balance', $amount);

        Log::info('Account debited', [
            'account_id' => $account->id,
            'account_number' => $account->account_number,
            'amount' => $amount,
            'new_balance' => $account->fresh()->balance,
        ]);
    }

    /**
     * Handle status changes.
     */
    protected function handleStatusChange(Transaction $transaction, string $oldStatus, string $newStatus): void
    {
        // If moving from pending to completed, update balances
        if ($oldStatus === 'pending' && $newStatus === 'completed') {
            $this->updateAccountBalances($transaction);
        }

        // If moving from completed to any other status, reverse balances
        if ($oldStatus === 'completed' && $newStatus !== 'completed') {
            $this->reverseAccountBalances($transaction);
        }
    }

    /**
     * Recalculate account balances when amount changes.
     */
    protected function recalculateAccountBalances(Transaction $transaction, float $oldAmount, float $newAmount): void
    {
        $difference = $newAmount - $oldAmount;

        if ($difference == 0) {
            return;
        }

        switch ($transaction->type) {
            case 'deposit':
                if ($transaction->to_account_id) {
                    if ($difference > 0) {
                        $this->creditAccount($transaction->toAccount, $difference);
                    } else {
                        $this->debitAccount($transaction->toAccount, abs($difference));
                    }
                }
                break;

            case 'withdrawal':
                if ($transaction->from_account_id) {
                    if ($difference > 0) {
                        $this->debitAccount($transaction->fromAccount, $difference);
                    } else {
                        $this->creditAccount($transaction->fromAccount, abs($difference));
                    }
                }
                break;

            case 'transfer':
                if ($transaction->from_account_id && $transaction->to_account_id) {
                    if ($difference > 0) {
                        $this->debitAccount($transaction->fromAccount, $difference);
                        $this->creditAccount($transaction->toAccount, $difference);
                    } else {
                        $this->creditAccount($transaction->fromAccount, abs($difference));
                        $this->debitAccount($transaction->toAccount, abs($difference));
                    }
                }
                break;
        }
    }

    /**
     * Get transaction statistics.
     */
    public function getTransactionStatistics(array $filters = []): array
    {
        $query = Transaction::query();

        // Apply filters
        if (isset($filters['date_from'])) {
            $query->whereDate('created_at', '>=', $filters['date_from']);
        }

        if (isset($filters['date_to'])) {
            $query->whereDate('created_at', '<=', $filters['date_to']);
        }

        if (isset($filters['type'])) {
            $query->where('type', $filters['type']);
        }

        if (isset($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (isset($filters['currency'])) {
            $query->where('currency', $filters['currency']);
        }

        $totalTransactions = $query->count();
        $totalAmount = $query->sum('amount');
        $averageAmount = $totalTransactions > 0 ? $totalAmount / $totalTransactions : 0;

        $statusBreakdown = $query->selectRaw('status, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('status')
            ->get()
            ->keyBy('status');

        $typeBreakdown = $query->selectRaw('type, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('type')
            ->get()
            ->keyBy('type');

        $currencyBreakdown = $query->selectRaw('currency, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('currency')
            ->get()
            ->keyBy('currency');

        return [
            'total_transactions' => $totalTransactions,
            'total_amount' => $totalAmount,
            'average_amount' => $averageAmount,
            'status_breakdown' => $statusBreakdown,
            'type_breakdown' => $typeBreakdown,
            'currency_breakdown' => $currencyBreakdown,
        ];
    }

    /**
     * Format currency amount.
     */
    public function formatAmount(float $amount, string $currency): string
    {
        try {
            return Money::of($amount, $currency)->format();
        } catch (\Exception $e) {
            return $currency . number_format($amount, 2);
        }
    }

    /**
     * Validate transaction data.
     */
    public function validateTransactionData(array $data): array
    {
        $errors = [];

        // Check if user exists
        if (!User::find($data['user_id'])) {
            $errors[] = 'User not found.';
        }

        // Check if accounts exist
        if (isset($data['from_account_id']) && !Account::find($data['from_account_id'])) {
            $errors[] = 'From account not found.';
        }

        if (isset($data['to_account_id']) && !Account::find($data['to_account_id'])) {
            $errors[] = 'To account not found.';
        }

        // Check account ownership for transfers
        if ($data['type'] === 'transfer' && isset($data['from_account_id'])) {
            $fromAccount = Account::find($data['from_account_id']);
            if ($fromAccount && $fromAccount->user_id !== $data['user_id']) {
                $errors[] = 'User does not own the from account.';
            }
        }

        // Check sufficient funds for withdrawals and transfers
        if (in_array($data['type'], ['withdrawal', 'transfer']) && isset($data['from_account_id'])) {
            $fromAccount = Account::find($data['from_account_id']);
            if ($fromAccount && $fromAccount->available_balance < $data['amount']) {
                $errors[] = 'Insufficient funds in the from account.';
            }
        }

        return $errors;
    }
}
