<?php

namespace App\Services;

use App\Models\Transaction;
use App\Models\Account;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class TransactionGeneratorService
{
    /**
     * Generate transactions for an account within a date range
     */
    public function generateTransactions(
        int $accountId,
        Carbon $startDate,
        Carbon $endDate,
        int $transactionCount = 10,
        string $direction = 'mixed' // 'incoming', 'outgoing', 'mixed'
    ): array {
        $account = Account::findOrFail($accountId);
        $user = $account->user;
        
        $transactions = [];
        $currentDate = $startDate->copy();
        
        // Transaction types to exclude (as requested)
        $excludedTypes = ['card_payment', 'atm'];
        $excludedSubcategories = ['inter_account', 'cheque_deposit'];
        
        // Available transaction types
        $transactionTypes = [
            'deposit' => [
                'subcategories' => ['wire_deposit', 'ach_deposit', 'bank_transfer_deposit', 'digital_wallet_deposit', 'interest_deposit', 'refund_deposit'],
                'descriptions' => [
                    'Wire transfer deposit',
                    'ACH deposit from external bank',
                    'Bank transfer deposit',
                    'Digital wallet deposit',
                    'Interest payment',
                    'Refund deposit'
                ]
            ],
            'withdrawal' => [
                'subcategories' => ['wire_withdrawal', 'ach_withdrawal', 'bank_transfer_withdrawal', 'digital_wallet_withdrawal'],
                'descriptions' => [
                    'Wire transfer withdrawal',
                    'ACH withdrawal to external bank',
                    'Bank transfer withdrawal',
                    'Digital wallet withdrawal'
                ]
            ],
            'transfer' => [
                'subcategories' => ['wire_transfer', 'ach_transfer', 'swift_transfer', 'sepa_transfer', 'domestic_transfer'],
                'descriptions' => [
                    'Wire transfer to external account',
                    'ACH transfer to external account',
                    'SWIFT international transfer',
                    'SEPA transfer',
                    'Domestic bank transfer'
                ]
            ],
            'payment' => [
                'subcategories' => ['bill_payment', 'merchant_payment', 'utility_payment', 'loan_payment'],
                'descriptions' => [
                    'Bill payment',
                    'Merchant payment',
                    'Utility payment',
                    'Loan payment'
                ]
            ],
            'refund' => [
                'subcategories' => ['merchant_refund', 'service_refund', 'fee_refund'],
                'descriptions' => [
                    'Merchant refund',
                    'Service refund',
                    'Fee refund'
                ]
            ],
            'fee' => [
                'subcategories' => ['maintenance_fee', 'transfer_fee', 'overdraft_fee', 'late_fee'],
                'descriptions' => [
                    'Monthly maintenance fee',
                    'Transfer processing fee',
                    'Overdraft fee',
                    'Late payment fee'
                ]
            ],
            'interest' => [
                'subcategories' => ['savings_interest', 'deposit_interest', 'loan_interest'],
                'descriptions' => [
                    'Savings account interest',
                    'Deposit interest payment',
                    'Loan interest payment'
                ]
            ]
        ];
        
        // Generate random dates within the range
        $dateRange = $endDate->diffInDays($startDate);
        $randomDates = [];
        
        for ($i = 0; $i < $transactionCount; $i++) {
            $randomDays = rand(0, $dateRange);
            $randomHours = rand(0, 23);
            $randomMinutes = rand(0, 59);
            
            $randomDates[] = $startDate->copy()->addDays($randomDays)
                ->setHour($randomHours)
                ->setMinute($randomMinutes)
                ->setSecond(0);
        }
        
        // Sort dates
        sort($randomDates);
        
        foreach ($randomDates as $transactionDate) {
            $type = array_rand($transactionTypes);
            $typeData = $transactionTypes[$type];
            $subcategory = $typeData['subcategories'][array_rand($typeData['subcategories'])];
            $description = $typeData['descriptions'][array_rand($typeData['descriptions'])];
            
            // Determine if this should be incoming or outgoing
            $isIncoming = $this->shouldBeIncoming($type, $direction);
            
            // Generate transaction data
            $transactionData = $this->generateTransactionData(
                $account,
                $user,
                $type,
                $subcategory,
                $description,
                $transactionDate,
                $isIncoming
            );
            
            // Create the transaction
            $transaction = $this->createTransaction($transactionData);
            $transactions[] = $transaction;
        }
        
        return $transactions;
    }
    
    /**
     * Determine if transaction should be incoming based on type and direction
     */
    private function shouldBeIncoming(string $type, string $direction): bool
    {
        if ($direction === 'incoming') return true;
        if ($direction === 'outgoing') return false;
        
        // For mixed direction, determine based on type
        $incomingTypes = ['deposit', 'refund', 'interest'];
        $outgoingTypes = ['withdrawal', 'transfer', 'payment', 'fee'];
        
        if (in_array($type, $incomingTypes)) return true;
        if (in_array($type, $outgoingTypes)) return false;
        
        // Default to random for mixed
        return rand(0, 1) === 1;
    }
    
    /**
     * Generate transaction data
     */
    private function generateTransactionData(
        Account $account,
        User $user,
        string $type,
        string $subcategory,
        string $description,
        Carbon $transactionDate,
        bool $isIncoming
    ): array {
        // Generate amounts based on type
        $amount = $this->generateAmount($type, $isIncoming);
        
        // Determine accounts
        $fromAccountId = $isIncoming ? null : $account->id;
        $toAccountId = $isIncoming ? $account->id : null;
        
        // Generate external account for outgoing transactions
        if (!$isIncoming && in_array($type, ['transfer', 'withdrawal', 'payment'])) {
            $toAccountId = $this->generateExternalAccountId();
        }
        
        // Generate fees
        $feeAmount = $this->generateFee($type, $amount);
        $taxAmount = $this->generateTax($amount);
        $netAmount = $amount - $feeAmount - $taxAmount;
        
        // Generate transfer method for transfers
        $transferMethod = null;
        $transferSpeed = null;
        if ($type === 'transfer') {
            $transferMethods = ['wire', 'ach', 'swift', 'sepa', 'domestic'];
            $transferMethod = $transferMethods[array_rand($transferMethods)];
            
            $speeds = ['instant', 'same_day', 'next_day', 'standard'];
            $transferSpeed = $speeds[array_rand($speeds)];
        }
        
        // Generate payment method for payments
        $paymentMethod = null;
        if ($type === 'payment') {
            $paymentMethods = ['bank_transfer', 'digital_wallet', 'wire', 'ach'];
            $paymentMethod = $paymentMethods[array_rand($paymentMethods)];
        }
        
        return [
            'transaction_id' => 'TXN-' . $transactionDate->format('YmdHis') . '-' . strtoupper(Str::random(6)),
            'reference_number' => 'REF-' . $transactionDate->format('YmdHis') . '-' . strtoupper(Str::random(4)),
            'external_reference' => 'EXT-' . strtoupper(Str::random(8)),
            'user_id' => $user->id,
            'from_account_id' => $fromAccountId,
            'to_account_id' => $toAccountId,
            'type' => $type,
            'category' => 'banking',
            'subcategory' => $subcategory,
            'amount' => $amount,
            'currency' => $account->currency,
            'exchange_rate' => 1.0, // Assuming same currency
            'converted_amount' => $amount,
            'fee_amount' => $feeAmount,
            'tax_amount' => $taxAmount,
            'net_amount' => $netAmount,
            'description' => $description,
            'notes' => $this->generateNotes($type, $isIncoming),
            'status' => 'completed',
            'verification_status' => 'verified',
            'transfer_method' => $transferMethod,
            'transfer_speed' => $transferSpeed,
            'payment_method' => $paymentMethod,
            'ip_address' => $this->generateIpAddress(),
            'risk_level' => 'low',
            'created_at' => $transactionDate,
            'updated_at' => $transactionDate,
            'processed_at' => $transactionDate,
            'completed_at' => $transactionDate,
        ];
    }
    
    /**
     * Generate appropriate amount based on type and direction
     */
    private function generateAmount(string $type, bool $isIncoming): float
    {
        $ranges = [
            'deposit' => [100, 50000],
            'withdrawal' => [50, 10000],
            'transfer' => [25, 25000],
            'payment' => [10, 5000],
            'refund' => [5, 2000],
            'fee' => [5, 100],
            'interest' => [1, 500],
        ];
        
        $range = $ranges[$type] ?? [10, 1000];
        $min = $range[0];
        $max = $range[1];
        
        // Add some randomness with decimal places
        $amount = rand($min * 100, $max * 100) / 100;
        
        return round($amount, 2);
    }
    
    /**
     * Generate fee amount
     */
    private function generateFee(string $type, float $amount): float
    {
        $feeRates = [
            'transfer' => 0.02, // 2%
            'withdrawal' => 0.015, // 1.5%
            'payment' => 0.01, // 1%
            'deposit' => 0.005, // 0.5%
        ];
        
        $rate = $feeRates[$type] ?? 0.01;
        $fee = $amount * $rate;
        
        // Minimum fee
        $minFee = 2.50;
        $fee = max($fee, $minFee);
        
        return round($fee, 2);
    }
    
    /**
     * Generate tax amount
     */
    private function generateTax(float $amount): float
    {
        $taxRate = 0.05; // 5% tax
        $tax = $amount * $taxRate;
        return round($tax, 2);
    }
    
    /**
     * Generate notes
     */
    private function generateNotes(string $type, bool $isIncoming): string
    {
        $notes = [
            'deposit' => $isIncoming ? 'Deposit received successfully' : 'Deposit processed',
            'withdrawal' => $isIncoming ? 'Withdrawal request received' : 'Withdrawal processed',
            'transfer' => $isIncoming ? 'Transfer received from external account' : 'Transfer sent to external account',
            'payment' => $isIncoming ? 'Payment received' : 'Payment processed',
            'refund' => $isIncoming ? 'Refund received' : 'Refund processed',
            'fee' => 'Service fee applied',
            'interest' => 'Interest payment applied',
        ];
        
        return $notes[$type] ?? 'Transaction processed';
    }
    
    /**
     * Generate IP address
     */
    private function generateIpAddress(): string
    {
        return rand(1, 255) . '.' . rand(1, 255) . '.' . rand(1, 255) . '.' . rand(1, 255);
    }
    
    /**
     * Generate external account ID (simulated)
     */
    private function generateExternalAccountId(): int
    {
        // Return a random account ID from existing accounts
        $account = Account::inRandomOrder()->first();
        return $account ? $account->id : 1;
    }
    
    /**
     * Create transaction in database
     */
    private function createTransaction(array $data): Transaction
    {
        return DB::transaction(function () use ($data) {
            $transaction = Transaction::create($data);
            
            // Update account balances if transaction is completed
            if ($data['status'] === 'completed') {
                $this->updateAccountBalances($transaction);
            }
            
            return $transaction;
        });
    }
    
    /**
     * Update account balances
     */
    private function updateAccountBalances(Transaction $transaction): void
    {
        if ($transaction->from_account_id) {
            $fromAccount = Account::find($transaction->from_account_id);
            if ($fromAccount) {
                $fromAccount->decrement('balance', $transaction->amount);
                $fromAccount->decrement('available_balance', $transaction->amount);
            }
        }
        
        if ($transaction->to_account_id) {
            $toAccount = Account::find($transaction->to_account_id);
            if ($toAccount) {
                $toAccount->increment('balance', $transaction->net_amount);
                $toAccount->increment('available_balance', $transaction->net_amount);
            }
        }
    }
    
    /**
     * Generate transactions for multiple accounts
     */
    public function generateBulkTransactions(
        array $accountIds,
        Carbon $startDate,
        Carbon $endDate,
        int $transactionsPerAccount = 10,
        string $direction = 'mixed'
    ): array {
        $allTransactions = [];
        
        foreach ($accountIds as $accountId) {
            $transactions = $this->generateTransactions(
                $accountId,
                $startDate,
                $endDate,
                $transactionsPerAccount,
                $direction
            );
            
            $allTransactions = array_merge($allTransactions, $transactions);
        }
        
        return $allTransactions;
    }
}
