<?php

namespace App\Services;

use App\Models\Transaction;
use App\Models\Account;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DepositService
{
    /**
     * Create a new deposit transaction
     */
    public function createDeposit(array $data): Transaction
    {
        return DB::transaction(function () use ($data) {
            $transaction = Transaction::create($data);

            // Log the deposit creation
            Log::info('Deposit created', [
                'transaction_id' => $transaction->transaction_id,
                'user_id' => $transaction->user_id,
                'amount' => $transaction->amount,
                'type' => $transaction->subcategory,
            ]);

            return $transaction;
        });
    }

    /**
     * Process a pending deposit
     */
    public function processDeposit(Transaction $transaction): bool
    {
        if ($transaction->status !== 'pending') {
            return false;
        }

        return DB::transaction(function () use ($transaction) {
            // Update transaction status
            $transaction->update([
                'status' => 'processing',
                'processed_at' => now(),
            ]);

            // Process the transaction
            $transaction->process();

            // Update account balance
            $account = $transaction->toAccount;
            if ($account) {
                $account->updateBalance($transaction->amount, 'credit');
            }

            // Complete the transaction
            $transaction->complete();

            Log::info('Deposit processed', [
                'transaction_id' => $transaction->transaction_id,
                'amount' => $transaction->amount,
                'account_id' => $transaction->to_account_id,
            ]);

            return true;
        });
    }

    /**
     * Reject a pending deposit
     */
    public function rejectDeposit(Transaction $transaction, string $reason): bool
    {
        if ($transaction->status !== 'pending') {
            return false;
        }

        return DB::transaction(function () use ($transaction, $reason) {
            $transaction->update([
                'status' => 'failed',
                'verification_status' => 'rejected',
                'notes' => $reason,
            ]);

            Log::info('Deposit rejected', [
                'transaction_id' => $transaction->transaction_id,
                'reason' => $reason,
            ]);

            return true;
        });
    }

    /**
     * Get deposit statistics for a user
     */
    public function getUserDepositStats(User $user): array
    {
        $baseQuery = Transaction::where('user_id', $user->id)
            ->where('type', 'deposit');

        return [
            'total_deposits' => $baseQuery->count(),
            'pending_deposits' => $baseQuery->where('status', 'pending')->count(),
            'completed_deposits' => $baseQuery->where('status', 'completed')->count(),
            'total_amount' => $baseQuery->where('status', 'completed')->sum('amount'),
            'pending_amount' => $baseQuery->where('status', 'pending')->sum('amount'),
            'this_month' => $baseQuery->whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->where('status', 'completed')
                ->sum('amount'),
        ];
    }

    /**
     * Get recent deposits for a user
     */
    public function getRecentDeposits(User $user, int $limit = 10)
    {
        return Transaction::with(['toAccount'])
            ->where('user_id', $user->id)
            ->where('type', 'deposit')
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    /**
     * Check if user can make a deposit
     */
    public function canMakeDeposit(User $user, float $amount): array
    {
        $errors = [];

        // Check if user has active accounts
        $activeAccounts = $user->accounts()
            ->where('status', 'active')
            ->where('verification_status', 'verified')
            ->count();

        if ($activeAccounts === 0) {
            $errors[] = 'No active accounts available for deposits';
        }

        // Check daily deposit limit
        $todayDeposits = Transaction::where('user_id', $user->id)
            ->where('type', 'deposit')
            ->whereDate('created_at', today())
            ->sum('amount');

        $dailyLimit = 100000; // $100,000 daily limit
        if (($todayDeposits + $amount) > $dailyLimit) {
            $errors[] = 'Daily deposit limit exceeded';
        }

        // Check monthly deposit limit
        $monthlyDeposits = Transaction::where('user_id', $user->id)
            ->where('type', 'deposit')
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('amount');

        $monthlyLimit = 500000; // $500,000 monthly limit
        if (($monthlyDeposits + $amount) > $monthlyLimit) {
            $errors[] = 'Monthly deposit limit exceeded';
        }

        return [
            'can_deposit' => empty($errors),
            'errors' => $errors,
        ];
    }
}
