<?php

namespace App\Services;

use App\Models\ComplianceLog;
use App\Models\User;
use App\Models\Account;
use App\Models\Transaction;
use App\Models\KycDocument;
use App\Helpers\SettingsHelper;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class ComplianceService
{
    /**
     * Create a compliance log entry
     */
    public function createLog(array $data): ComplianceLog
    {
        $log = ComplianceLog::create(array_merge([
            'source_ip' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ], $data));

        Log::info('Compliance log created', [
            'log_id' => $log->id,
            'type' => $log->type,
            'severity' => $log->severity,
            'user_id' => $log->user_id,
        ]);

        return $log;
    }

    /**
     * Flag a transaction for review
     */
    public function flagTransaction(Transaction $transaction, string $reason, string $severity = 'medium'): ComplianceLog
    {
        // Load the account relationship if not already loaded
        if (!$transaction->relationLoaded('fromAccount')) {
            $transaction->load('fromAccount');
        }

        return $this->createLog([
            'type' => 'transaction_flag',
            'category' => $this->determineTransactionCategory($transaction),
            'severity' => $severity,
            'description' => $reason,
            'user_id' => $transaction->user_id,
            'account_id' => $transaction->from_account_id,
            'transaction_id' => $transaction->id,
            'amount' => $transaction->amount,
            'currency' => $transaction->currency,
            'details' => [
                'transaction_type' => $transaction->type,
                'transaction_reference' => $transaction->reference_number,
                'flag_reason' => $reason,
            ],
            'risk_factors' => $this->assessTransactionRisk($transaction),
        ]);
    }

    /**
     * Flag a KYC document for review
     */
    public function flagKycDocument(KycDocument $document, string $reason, string $severity = 'medium'): ComplianceLog
    {
        return $this->createLog([
            'type' => 'kyc_review',
            'category' => 'document_verification',
            'severity' => $severity,
            'description' => $reason,
            'user_id' => $document->user_id,
            'kyc_document_id' => $document->id,
            'details' => [
                'document_type' => $document->document_type,
                'document_name' => $document->document_name,
                'file_name' => $document->file_name,
                'review_reason' => $reason,
            ],
        ]);
    }

    /**
     * Perform risk assessment on a user
     */
    public function assessUserRisk(User $user): ComplianceLog
    {
        $riskFactors = $this->calculateUserRiskFactors($user);
        $severity = $this->determineRiskSeverity($riskFactors);

        return $this->createLog([
            'type' => 'risk_assessment',
            'category' => 'unusual_pattern',
            'severity' => $severity,
            'description' => 'Automated risk assessment for user account',
            'user_id' => $user->id,
            'details' => [
                'assessment_date' => now()->toISOString(),
                'risk_score' => $this->calculateRiskScore($riskFactors),
            ],
            'risk_factors' => $riskFactors,
        ]);
    }

    /**
     * Update compliance log status
     */
    public function updateLogStatus(ComplianceLog $log, string $status, ?string $notes = null, ?string $resolutionStatus = null): ComplianceLog
    {
        $log->update([
            'status' => $status,
            'reviewed_by' => Auth::id(),
            'reviewed_at' => now(),
            'admin_notes' => $notes,
            'resolution_status' => $resolutionStatus,
        ]);

        Log::info('Compliance log status updated', [
            'log_id' => $log->id,
            'new_status' => $status,
            'reviewed_by' => Auth::id(),
        ]);

        return $log;
    }

    /**
     * Get pending compliance logs
     */
    public function getPendingLogs(int $limit = 50): \Illuminate\Database\Eloquent\Collection
    {
        return ComplianceLog::with(['user', 'account', 'transaction', 'kycDocument'])
            ->where('status', 'pending')
            ->orderBy('severity', 'desc')
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    /**
     * Get high priority logs
     */
    public function getHighPriorityLogs(int $limit = 20): \Illuminate\Database\Eloquent\Collection
    {
        return ComplianceLog::with(['user', 'account', 'transaction', 'kycDocument'])
            ->whereIn('severity', ['high', 'critical'])
            ->whereIn('status', ['pending', 'under_review'])
            ->orderBy('severity', 'desc')
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    /**
     * Get compliance statistics
     */
    public function getComplianceStats(): array
    {
        return [
            'pending' => ComplianceLog::where('status', 'pending')->count(),
            'under_review' => ComplianceLog::where('status', 'under_review')->count(),
            'high_priority' => ComplianceLog::whereIn('severity', ['high', 'critical'])
                ->whereIn('status', ['pending', 'under_review'])
                ->count(),
            'resolved_today' => ComplianceLog::whereIn('status', ['approved', 'rejected'])
                ->whereDate('reviewed_at', today())
                ->count(),
            'total_this_month' => ComplianceLog::whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->count(),
        ];
    }

    /**
     * Determine transaction category based on transaction details
     */
    private function determineTransactionCategory(Transaction $transaction): string
    {
        // High value transaction threshold
        $highValueThreshold = 10000; // $10,000

        if ($transaction->amount >= $highValueThreshold) {
            return 'high_value_transaction';
        }

        // Check for suspicious patterns
        if ($this->isSuspiciousTransaction($transaction)) {
            return 'suspicious_activity';
        }

        return 'unusual_pattern';
    }

    /**
     * Check if transaction is suspicious
     */
    private function isSuspiciousTransaction(Transaction $transaction): bool
    {
        // Check for round numbers (potential structuring)
        if ($transaction->amount % 1000 === 0 && $transaction->amount >= 5000) {
            return true;
        }

        // Check for rapid successive transactions
        $recentTransactions = Transaction::where('from_account_id', $transaction->from_account_id)
            ->where('created_at', '>=', now()->subHours(1))
            ->count();

        if ($recentTransactions >= 5) {
            return true;
        }

        return false;
    }

    /**
     * Assess transaction risk factors
     */
    private function assessTransactionRisk(Transaction $transaction): array
    {
        $riskFactors = [];

        // High value risk
        if ($transaction->amount >= 10000) {
            $riskFactors[] = 'high_value_transaction';
        }

        // Round number risk
        if ($transaction->amount % 1000 === 0) {
            $riskFactors[] = 'round_number_amount';
        }

        // Time-based risk (transactions outside business hours)
        $hour = now()->hour;
        if ($hour < 6 || $hour > 22) {
            $riskFactors[] = 'off_hours_transaction';
        }

        // International transfer risk
        if ($transaction->type === 'international_transfer') {
            $riskFactors[] = 'international_transfer';
        }

        return $riskFactors;
    }

    /**
     * Calculate user risk factors
     */
    private function calculateUserRiskFactors(User $user): array
    {
        $riskFactors = [];

        // Check account age
        $accountAge = $user->created_at->diffInDays(now());
        if ($accountAge < 30) {
            $riskFactors[] = 'new_account';
        }

        // Check transaction volume
        $transactionCount = Transaction::where('user_id', $user->id)->count();

        if ($transactionCount === 0) {
            $riskFactors[] = 'no_transaction_history';
        }

        // Check KYC status
        if (!$user->profile || $user->profile->kyc_status !== 'verified') {
            $riskFactors[] = 'incomplete_kyc';
        }

        return $riskFactors;
    }

    /**
     * Determine risk severity based on factors
     */
    private function determineRiskSeverity(array $riskFactors): string
    {
        $criticalFactors = ['high_value_transaction', 'suspicious_activity'];
        $highFactors = ['international_transfer', 'new_account'];
        $mediumFactors = ['round_number_amount', 'off_hours_transaction'];

        if (array_intersect($criticalFactors, $riskFactors)) {
            return 'critical';
        }

        if (array_intersect($highFactors, $riskFactors)) {
            return 'high';
        }

        if (array_intersect($mediumFactors, $riskFactors)) {
            return 'medium';
        }

        return 'low';
    }

    /**
     * Calculate overall risk score
     */
    private function calculateRiskScore(array $riskFactors): int
    {
        $score = 0;
        $factorWeights = [
            'high_value_transaction' => 30,
            'suspicious_activity' => 25,
            'international_transfer' => 20,
            'new_account' => 15,
            'round_number_amount' => 10,
            'off_hours_transaction' => 5,
            'incomplete_kyc' => 15,
            'no_transaction_history' => 10,
        ];

        foreach ($riskFactors as $factor) {
            $score += $factorWeights[$factor] ?? 5;
        }

        return min($score, 100); // Cap at 100
    }

    /**
     * Monitor transaction for AML compliance
     */
    public function monitorTransaction(Transaction $transaction): ?ComplianceLog
    {
        // Check if AML monitoring is enabled
        if (!SettingsHelper::isAmlMonitoringEnabled()) {
            return null;
        }

        $suspiciousThreshold = SettingsHelper::getSuspiciousActivityThreshold();
        $riskFactors = [];
        $severity = 'low';

        // Check suspicious activity threshold
        if ($transaction->amount >= $suspiciousThreshold) {
            $riskFactors[] = 'exceeds_suspicious_threshold';
            $severity = 'high';
        }

        // Check for structuring (multiple transactions just under threshold)
        if ($this->detectStructuring($transaction)) {
            $riskFactors[] = 'potential_structuring';
            $severity = 'critical';
        }

        // Check for unusual patterns
        if ($this->detectUnusualPatterns($transaction)) {
            $riskFactors[] = 'unusual_transaction_pattern';
            $severity = $severity === 'critical' ? 'critical' : 'medium';
        }

        // Check for high-risk countries
        if ($this->isHighRiskCountry($transaction)) {
            $riskFactors[] = 'high_risk_country';
            $severity = $severity === 'critical' ? 'critical' : 'high';
        }

        // Check for PEP (Politically Exposed Person) transactions
        if ($this->isPepTransaction($transaction)) {
            $riskFactors[] = 'pep_transaction';
            $severity = 'critical';
        }

        // Only create log if there are risk factors
        if (!empty($riskFactors)) {
            return $this->createLog([
                'type' => 'aml_monitoring',
                'category' => 'transaction_monitoring',
                'severity' => $severity,
                'description' => 'Transaction flagged for AML review',
                'user_id' => $transaction->user_id,
                'account_id' => $transaction->from_account_id,
                'transaction_id' => $transaction->id,
                'amount' => $transaction->amount,
                'currency' => $transaction->currency,
                'details' => [
                    'transaction_type' => $transaction->type,
                    'transaction_reference' => $transaction->reference_number,
                    'suspicious_threshold' => $suspiciousThreshold,
                    'risk_factors' => $riskFactors,
                ],
                'risk_factors' => $riskFactors,
            ]);
        }

        return null;
    }

    /**
     * Monitor user for AML compliance
     */
    public function monitorUser(User $user): ?ComplianceLog
    {
        // Check if AML monitoring is enabled
        if (!SettingsHelper::isAmlMonitoringEnabled()) {
            return null;
        }

        $riskFactors = [];
        $severity = 'low';

        // Check for unusual transaction volume
        $dailyVolume = $this->getUserDailyTransactionVolume($user);
        $monthlyVolume = $this->getUserMonthlyTransactionVolume($user);
        $suspiciousThreshold = SettingsHelper::getSuspiciousActivityThreshold();

        if ($dailyVolume >= $suspiciousThreshold) {
            $riskFactors[] = 'high_daily_volume';
            $severity = 'high';
        }

        if ($monthlyVolume >= ($suspiciousThreshold * 10)) {
            $riskFactors[] = 'high_monthly_volume';
            $severity = $severity === 'high' ? 'critical' : 'high';
        }

        // Check for rapid successive transactions
        if ($this->hasRapidTransactions($user)) {
            $riskFactors[] = 'rapid_successive_transactions';
            $severity = $severity === 'critical' ? 'critical' : 'medium';
        }

        // Check for round number patterns
        if ($this->hasRoundNumberPattern($user)) {
            $riskFactors[] = 'round_number_pattern';
            $severity = $severity === 'critical' ? 'critical' : 'medium';
        }

        // Only create log if there are risk factors
        if (!empty($riskFactors)) {
            return $this->createLog([
                'type' => 'aml_monitoring',
                'category' => 'user_monitoring',
                'severity' => $severity,
                'description' => 'User flagged for AML review',
                'user_id' => $user->id,
                'details' => [
                    'daily_volume' => $dailyVolume,
                    'monthly_volume' => $monthlyVolume,
                    'suspicious_threshold' => $suspiciousThreshold,
                    'risk_factors' => $riskFactors,
                ],
                'risk_factors' => $riskFactors,
            ]);
        }

        return null;
    }

    /**
     * Detect potential structuring (multiple transactions just under threshold)
     */
    private function detectStructuring(Transaction $transaction): bool
    {
        $suspiciousThreshold = SettingsHelper::getSuspiciousActivityThreshold();
        $threshold = $suspiciousThreshold * 0.9; // 90% of threshold

        // Check if transaction is just under threshold
        if ($transaction->amount >= $threshold && $transaction->amount < $suspiciousThreshold) {
            // Check for similar transactions in the last 24 hours
            $similarTransactions = Transaction::where('from_account_id', $transaction->from_account_id)
                ->where('amount', '>=', $threshold)
                ->where('amount', '<', $suspiciousThreshold)
                ->where('created_at', '>=', now()->subDay())
                ->count();

            return $similarTransactions >= 3; // 3 or more similar transactions
        }

        return false;
    }

    /**
     * Detect unusual transaction patterns
     */
    private function detectUnusualPatterns(Transaction $transaction): bool
    {
        // Check for transactions at unusual times
        $hour = $transaction->created_at->hour;
        if ($hour < 6 || $hour > 22) {
            return true;
        }

        // Check for weekend transactions (high value)
        if ($transaction->created_at->isWeekend() && $transaction->amount >= 5000) {
            return true;
        }

        // Check for rapid successive transactions
        $recentCount = Transaction::where('from_account_id', $transaction->from_account_id)
            ->where('created_at', '>=', now()->subMinutes(30))
            ->count();

        return $recentCount >= 5;
    }

    /**
     * Check if transaction involves high-risk country
     */
    private function isHighRiskCountry(Transaction $transaction): bool
    {
        // This would typically check against a list of high-risk countries
        // For now, we'll use a simple check based on transaction metadata
        $highRiskCountries = ['AF', 'IR', 'KP', 'SY']; // Example high-risk country codes

        if (isset($transaction->metadata['beneficiary_country'])) {
            return in_array($transaction->metadata['beneficiary_country'], $highRiskCountries);
        }

        return false;
    }

    /**
     * Check if transaction involves PEP
     */
    private function isPepTransaction(Transaction $transaction): bool
    {
        // This would typically check against a PEP database
        // For now, we'll use a simple check based on user profile
        $user = $transaction->user;
        if ($user->profile && isset($user->profile->metadata['is_pep'])) {
            return $user->profile->metadata['is_pep'] === true;
        }

        return false;
    }

    /**
     * Get user's daily transaction volume
     */
    private function getUserDailyTransactionVolume(User $user): float
    {
        return Transaction::where('user_id', $user->id)
            ->whereDate('created_at', today())
            ->sum('amount');
    }

    /**
     * Get user's monthly transaction volume
     */
    private function getUserMonthlyTransactionVolume(User $user): float
    {
        return Transaction::where('user_id', $user->id)
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('amount');
    }

    /**
     * Check if user has rapid successive transactions
     */
    private function hasRapidTransactions(User $user): bool
    {
        $rapidCount = Transaction::where('user_id', $user->id)
            ->where('created_at', '>=', now()->subHour())
            ->count();

        return $rapidCount >= 10;
    }

    /**
     * Check if user has round number pattern
     */
    private function hasRoundNumberPattern(User $user): bool
    {
        $roundTransactions = Transaction::where('user_id', $user->id)
            ->where('amount', '>=', 1000)
            ->whereRaw('amount % 1000 = 0')
            ->where('created_at', '>=', now()->subWeek())
            ->count();

        return $roundTransactions >= 5;
    }

    /**
     * Get AML monitoring statistics
     */
    public function getAmlStats(): array
    {
        $cacheKey = 'aml_stats_' . now()->format('Y-m-d');
        
        return Cache::remember($cacheKey, 300, function () { // 5 minutes cache
            return [
                'total_monitored_transactions' => Transaction::where('created_at', '>=', now()->subDay())->count(),
                'flagged_transactions' => ComplianceLog::where('type', 'aml_monitoring')
                    ->where('created_at', '>=', now()->subDay())
                    ->count(),
                'high_risk_transactions' => ComplianceLog::where('type', 'aml_monitoring')
                    ->whereIn('severity', ['high', 'critical'])
                    ->where('created_at', '>=', now()->subDay())
                    ->count(),
                'structuring_detected' => ComplianceLog::where('type', 'aml_monitoring')
                    ->whereJsonContains('risk_factors', 'potential_structuring')
                    ->where('created_at', '>=', now()->subDay())
                    ->count(),
                'pep_transactions' => ComplianceLog::where('type', 'aml_monitoring')
                    ->whereJsonContains('risk_factors', 'pep_transaction')
                    ->where('created_at', '>=', now()->subDay())
                    ->count(),
                'suspicious_threshold' => SettingsHelper::getSuspiciousActivityThreshold(),
            ];
        });
    }

    /**
     * Run automated AML monitoring for all recent transactions
     */
    public function runAutomatedMonitoring(): array
    {
        $results = [
            'transactions_monitored' => 0,
            'transactions_flagged' => 0,
            'users_monitored' => 0,
            'users_flagged' => 0,
        ];

        // Monitor recent transactions (last 24 hours)
        $recentTransactions = Transaction::where('created_at', '>=', now()->subDay())
            ->where('status', 'completed')
            ->get();

        foreach ($recentTransactions as $transaction) {
            $results['transactions_monitored']++;
            $log = $this->monitorTransaction($transaction);
            if ($log) {
                $results['transactions_flagged']++;
            }
        }

        // Monitor active users
        $activeUsers = User::whereHas('transactions', function ($query) {
            $query->where('created_at', '>=', now()->subDay());
        })->get();

        foreach ($activeUsers as $user) {
            $results['users_monitored']++;
            $log = $this->monitorUser($user);
            if ($log) {
                $results['users_flagged']++;
            }
        }

        Log::info('Automated AML monitoring completed', $results);
        return $results;
    }
}
