<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

class Country extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'countries';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'iso',
        'iso3',
        'dial',
        'currency',
        'currency_name',
        'currency_symbol',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'id' => 'integer',
        'name' => 'string',
        'iso' => 'string',
        'iso3' => 'string',
        'dial' => 'string',
        'currency' => 'string',
        'currency_name' => 'string',
        'currency_symbol' => 'string',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'created_at' => 'datetime',
            'updated_at' => 'datetime',
        ];
    }

    // Accessors & Mutators

    /**
     * Get the country's full name with ISO code.
     *
     * @return string
     */
    public function getFullNameAttribute(): string
    {
        return "{$this->name} ({$this->iso})";
    }

    /**
     * Get the formatted phone dial code.
     *
     * @return string
     */
    public function getFormattedDialAttribute(): string
    {
        return "+{$this->dial}";
    }

    /**
     * Get the currency information as a formatted string.
     *
     * @return string|null
     */
    public function getCurrencyInfoAttribute(): ?string
    {
        if (!$this->currency) {
            return null;
        }

        $symbol = $this->currency_symbol ? " ({$this->currency_symbol})" : '';
        return "{$this->currency_name} - {$this->currency}{$symbol}";
    }

    /**
     * Check if the country has a currency.
     *
     * @return bool
     */
    public function getHasCurrencyAttribute(): bool
    {
        return !empty($this->currency);
    }

    // Query Scopes

    /**
     * Scope a query to only include countries with currencies.
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeWithCurrency(Builder $query): Builder
    {
        return $query->whereNotNull('currency');
    }

    /**
     * Scope a query to only include countries without currencies.
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeWithoutCurrency(Builder $query): Builder
    {
        return $query->whereNull('currency');
    }

    /**
     * Scope a query to search by country name.
     *
     * @param Builder $query
     * @param string $name
     * @return Builder
     */
    public function scopeByName(Builder $query, string $name): Builder
    {
        return $query->where('name', 'like', "%{$name}%");
    }

    /**
     * Scope a query to find by ISO code.
     *
     * @param Builder $query
     * @param string $iso
     * @return Builder
     */
    public function scopeByIso(Builder $query, string $iso): Builder
    {
        return $query->where('iso', strtoupper($iso))
                    ->orWhere('iso3', strtoupper($iso));
    }

    /**
     * Scope a query to find by currency code.
     *
     * @param Builder $query
     * @param string $currency
     * @return Builder
     */
    public function scopeByCurrency(Builder $query, string $currency): Builder
    {
        return $query->where('currency', strtoupper($currency));
    }

    /**
     * Scope a query to find by dial code.
     *
     * @param Builder $query
     * @param string $dial
     * @return Builder
     */
    public function scopeByDial(Builder $query, string $dial): Builder
    {
        $cleanDial = ltrim($dial, '+');
        return $query->where('dial', $cleanDial);
    }

    // Static Methods

    /**
     * Get all unique currencies.
     *
     * @return \Illuminate\Support\Collection
     */
    public static function getAllCurrencies()
    {
        return static::whereNotNull('currency')
                    ->select('id', 'name', 'currency', 'currency_name', 'currency_symbol')
                    ->distinct()
                    ->orderBy('currency')
                    ->get();
    }

    /**
     * Get countries by continent (you might want to add a continent field).
     *
     * @param string $continent
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getByContinent(string $continent)
    {
        // This would require adding a continent field to your table
        // return static::where('continent', $continent)->get();
        
        // For now, return empty collection
        return collect();
    }

    /**
     * Find country by various identifiers.
     *
     * @param string $identifier
     * @return Country|null
     */
    public static function findByIdentifier(string $identifier): ?Country
    {
        return static::where('name', $identifier)
                    ->orWhere('iso', strtoupper($identifier))
                    ->orWhere('iso3', strtoupper($identifier))
                    ->first();
    }

    // Instance Methods

    /**
     * Check if this country uses the Euro currency.
     *
     * @return bool
     */
    public function usesEuro(): bool
    {
        return $this->currency === 'EUR';
    }

    /**
     * Check if this country uses the US Dollar.
     *
     * @return bool
     */
    public function usesUSD(): bool
    {
        return $this->currency === 'USD';
    }

    /**
     * Get other countries that use the same currency.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function countriesWithSameCurrency()
    {
        if (!$this->currency) {
            return collect();
        }

        return static::where('currency', $this->currency)
                    ->where('id', '!=', $this->id)
                    ->get();
    }

    /**
     * Format phone number with country dial code.
     *
     * @param string $phoneNumber
     * @return string
     */
    public function formatPhoneNumber(string $phoneNumber): string
    {
        $cleanNumber = preg_replace('/[^0-9]/', '', $phoneNumber);
        return "+{$this->dial} {$cleanNumber}";
    }

    /**
     * Get flag emoji (if you want to add this feature).
     *
     * @return string|null
     */
    public function getFlagEmoji(): ?string
    {
        if (strlen($this->iso) !== 2) {
            return null;
        }

        // Convert ISO code to flag emoji
        $firstLetter = ord($this->iso[0]) - ord('A') + 0x1F1E6;
        $secondLetter = ord($this->iso[1]) - ord('A') + 0x1F1E6;
        
        return mb_chr($firstLetter) . mb_chr($secondLetter);
    }

    /**
     * Get the country as an array for API responses.
     *
     * @return array
     */
    public function toApiArray(): array
    {
        return [
            'id' => $this->id,
            'name' => $this->name,
            'iso' => $this->iso,
            'iso3' => $this->iso3,
            'dial_code' => $this->formatted_dial,
            'currency' => [
                'code' => $this->currency,
                'name' => $this->currency_name,
                'symbol' => $this->currency_symbol,
            ],
            'flag_emoji' => $this->getFlagEmoji(),
        ];
    }

}