<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class Beneficiary extends Model implements Auditable
{
    use HasFactory, SoftDeletes, AuditableTrait, LogsActivity;

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'user_id',
        'beneficiary_id',
        'name',
        'type',
        'email',
        'phone',
        'address',
        'city',
        'state_province',
        'postal_code',
        'country',
        'bank_name',
        'bank_code',
        'account_number',
        'iban',
        'swift_code',
        'currency',
        'business_name',
        'business_type',
        'business_registration_number',
        'tax_identification_number',
        'website',
        'verification_status',
        'verification_level',
        'verified_at',
        'verified_by',
        'verification_notes',
        'risk_level',
        'aml_verified',
        'sanctions_verified',
        'last_risk_assessment',
        'risk_notes',
        'daily_transfer_limit',
        'monthly_transfer_limit',
        'single_transfer_limit',
        'transfer_methods',
        'preferred_transfer_method',
        'allowed_accounts',
        'active',
        'last_transfer_at',
        'total_transfers',
        'total_transfer_amount',
        'two_factor_required',
        'notification_enabled',
        'notification_preferences',
        'security_settings',
        'notes',
        'metadata',
        'relationship',
        'expires_at',
        'auto_renewal'
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'daily_transfer_limit' => 'decimal:4',
        'monthly_transfer_limit' => 'decimal:4',
        'single_transfer_limit' => 'decimal:4',
        'total_transfer_amount' => 'decimal:4',
        'verified_at' => 'datetime',
        'last_risk_assessment' => 'datetime',
        'last_transfer_at' => 'datetime',
        'expires_at' => 'datetime',
        'aml_verified' => 'boolean',
        'sanctions_verified' => 'boolean',
        'active' => 'boolean',
        'two_factor_required' => 'boolean',
        'notification_enabled' => 'boolean',
        'auto_renewal' => 'boolean',
        'allowed_accounts' => 'array',
        'notification_preferences' => 'array',
        'security_settings' => 'array',
        'metadata' => 'array'
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [
        'id',
        'user_id',
        'account_id',
        'bank_code',
        'routing_number',
        'swift_code',
        'iban'
    ];

    /**
     * The attributes that should be appended.
     */
    protected $appends = [
        'full_name',
        'verification_status_label',
        'is_verified_label',
        'can_receive_transfer',
        'daily_limit_remaining',
        'monthly_limit_remaining'
    ];

    /**
     * Get the user that owns the beneficiary.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the account associated with the beneficiary.
     */
    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    /**
     * Get the transactions sent to this beneficiary.
     */
    public function transactions(): HasMany
    {
        return $this->hasMany(Transaction::class);
    }

    /**
     * Get the full name of the beneficiary.
     */
    public function getFullNameAttribute(): string
    {
        return $this->name;
    }

    /**
     * Get the verification status label.
     */
    public function getVerificationStatusLabelAttribute(): string
    {
        return match($this->verification_status) {
            'pending' => 'Pending Verification',
            'verified' => 'Verified',
            'rejected' => 'Rejected',
            'suspended' => 'Suspended',
            default => 'Unknown'
        };
    }

    /**
     * Get the verification status as a label.
     */
    public function getIsVerifiedLabelAttribute(): string
    {
        return $this->verification_status === 'verified' ? 'Verified' : 'Not Verified';
    }

    /**
     * Check if beneficiary can receive transfers.
     */
    public function getCanReceiveTransferAttribute(): bool
    {
        return $this->verification_status === 'verified' && $this->active;
    }

    /**
     * Get remaining daily limit.
     */
    public function getDailyLimitRemainingAttribute(): ?float
    {
        if (!$this->daily_transfer_limit) {
            return null;
        }

        $todayTransfers = $this->transactions()
            ->where('type', 'transfer')
            ->whereDate('created_at', today())
            ->sum('amount');

        return max(0, $this->daily_transfer_limit - $todayTransfers);
    }

    /**
     * Get remaining monthly limit.
     */
    public function getMonthlyLimitRemainingAttribute(): ?float
    {
        if (!$this->monthly_transfer_limit) {
            return null;
        }

        $monthlyTransfers = $this->transactions()
            ->where('type', 'transfer')
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('amount');

        return max(0, $this->monthly_transfer_limit - $monthlyTransfers);
    }

    /**
     * Scope for verified beneficiaries.
     */
    public function scopeVerified($query)
    {
        return $query->where('verification_status', 'verified');
    }

    /**
     * Scope for beneficiaries by type.
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Scope for beneficiaries by country.
     */
    public function scopeInCountry($query, $country)
    {
        return $query->where('country', $country);
    }

    /**
     * Scope for beneficiaries by currency.
     */
    public function scopeInCurrency($query, $currency)
    {
        return $query->where('currency', $currency);
    }

    /**
     * Scope for beneficiaries by verification status.
     */
    public function scopeWithVerificationStatus($query, $status)
    {
        return $query->where('verification_status', $status);
    }

    /**
     * Scope for active beneficiaries.
     */
    public function scopeActive($query)
    {
        return $query->where('verification_status', 'verified');
    }

    /**
     * Scope for pending verification.
     */
    public function scopePendingVerification($query)
    {
        return $query->where('verification_status', 'pending');
    }

    /**
     * Scope for rejected beneficiaries.
     */
    public function scopeRejected($query)
    {
        return $query->where('verification_status', 'rejected');
    }

    /**
     * Check if beneficiary can receive transfer amount.
     */
    public function canReceiveTransfer(float $amount): bool
    {
        if (!$this->can_receive_transfer) {
            return false;
        }

        // Check daily limit
        if ($this->daily_transfer_limit && $this->daily_limit_remaining < $amount) {
            return false;
        }

        // Check monthly limit
        if ($this->monthly_transfer_limit && $this->monthly_limit_remaining < $amount) {
            return false;
        }

        return true;
    }

    /**
     * Mark beneficiary as verified.
     */
    public function markAsVerified(): bool
    {
        $this->verification_status = 'verified';
        $this->verified_at = now();
        
        if ($this->save()) {
            // Log the verification
            activity()
                ->performedOn($this)
                ->log("Beneficiary {$this->full_name} verified successfully");
            
            return true;
        }
        return false;
    }

    /**
     * Mark beneficiary as rejected.
     */
    public function markAsRejected(string $reason = null): bool
    {
        $this->verification_status = 'rejected';
        
        if ($this->save()) {
            // Log the rejection
            activity()
                ->performedOn($this)
                ->log("Beneficiary {$this->full_name} rejected: {$reason}");
            
            return true;
        }

        return false;
    }

    /**
     * Suspend beneficiary.
     */
    public function suspend(string $reason = null): bool
    {
        $this->verification_status = 'suspended';
        
        if ($this->save()) {
            // Log the suspension
            activity()
                ->performedOn($this)
                ->log("Beneficiary {$this->full_name} suspended: {$reason}");
            
            return true;
        }

        return false;
    }

    /**
     * Get transfer history for beneficiary.
     */
    public function getTransferHistory(string $period = 'month'): array
    {
        $query = $this->transactions()
            ->where('type', 'transfer')
            ->where('status', 'completed');

        switch ($period) {
            case 'week':
                $query->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]);
                break;
            case 'month':
                $query->whereMonth('created_at', now()->month)
                    ->whereYear('created_at', now()->year);
                break;
            case 'year':
                $query->whereYear('created_at', now()->year);
                break;
        }

        return [
            'count' => $query->count(),
            'total_amount' => $query->sum('amount'),
            'average_amount' => $query->avg('amount'),
            'last_transfer' => $query->latest()->first()
        ];
    }

    /**
     * Get validation rules.
     */
    public static function getValidationRules(): array
    {
        return [
            'type' => 'required|in:individual,business,domestic,international',
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:255',
            'bank_name' => 'required|string|max:255',
            'account_number' => 'required|string|max:255',
            'country' => 'required|string|max:255',
            'currency' => 'required|string|size:3',
            'daily_transfer_limit' => 'nullable|numeric|min:0',
            'monthly_transfer_limit' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string|max:1000'
        ];
    }

    /**
     * Activity log options.
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly([
                'type', 'name', 'email',
                'bank_name', 'country', 'currency', 'verification_status'
            ])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /**
     * Audit configuration.
     */
    protected $auditInclude = [
        'type', 'name', 'email',
        'bank_name', 'country', 'currency', 'verification_status'
    ];

    protected $auditEvents = [
        'created', 'updated', 'deleted', 'restored'
    ];
}
