<?php

namespace App\Livewire\User;

use App\Models\TwoFactorAuth;
use App\Mail\TwoFactorVerificationMail;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Rule;
use Flux;
use PragmaRX\Google2FA\Google2FA;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\Writer\PngWriter;
use Spatie\Activitylog\Models\Activity;

#[Layout('components.layouts.app')]
class SecuritySettings extends Component
{
    // Password Change
    #[Rule('required|current_password')]
    public $currentPassword = '';

    #[Rule('required|min:8')]
    public $newPassword = '';

    #[Rule('required|same:newPassword')]
    public $confirmPassword = '';

    // PIN Management
    #[Rule('required|string|size:6|regex:/^[0-9]+$/')]
    public $currentPin = '';

    #[Rule('required|string|size:6|regex:/^[0-9]+$/')]
    public $newPin = '';

    #[Rule('required|same:newPin')]
    public $confirmPin = '';

    // 2FA Settings
    public $twoFactorEnabled = false;
    public $twoFactorMethod = 'totp';
    public $showTwoFactorSetup = false;
    public $twoFactorSecret = '';
    public $twoFactorQrCode = '';  // Will store base64 encoded image string
    public $twoFactorVerificationCode = '';
    public $backupCodes = [];  // Changed from null to empty array

    // Security Log
    public $securityLogs = [];
    public $showSecurityLog = false;

    // Remove the $google2fa property - it cannot be serialized by Livewire

    public function mount()
    {
        $this->loadUserData();
    }

    public function loadUserData()
    {
        $user = auth()->user();
        
        // Load 2FA status
        $twoFactor = $user->twoFactorAuth;
        if ($twoFactor) {
            $this->twoFactorEnabled = $twoFactor->enabled;
            $this->twoFactorMethod = $twoFactor->method ?? 'totp';
        }
    }

    // Password Management
    public function changePassword()
    {
        $this->validate([
            'currentPassword' => 'required',
            'newPassword' => ['required', 'min:8', 'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]/'],
            'confirmPassword' => 'required|same:newPassword'
        ]);

        $user = auth()->user();
        
        if (!Hash::check($this->currentPassword, $user->password)) {
            $this->addError('currentPassword', 'Current password is incorrect.');
            return;
        }

        $user->update([
            'password' => Hash::make($this->newPassword),
            'password_changed_at' => now()
        ]);

        // Log the password change
        Activity::create([
            'log_name' => 'security',
            'description' => 'Password changed',
            'subject_type' => get_class($user),
            'subject_id' => $user->id,
            'causer_type' => get_class($user),
            'causer_id' => $user->id,
        ]);

        $this->reset(['currentPassword', 'newPassword', 'confirmPassword']);
        
        // Send notification
        $notificationService = app(\App\Services\NotificationService::class);
        $notificationService->notifyPasswordChanged($user);
        
        Flux::toast(
            variant: 'success',
            heading: 'Password Updated!',
            text: 'Your password has been changed successfully.'
        );
    }

    // PIN Management
    public function changePin()
    {
        $this->validate([
            'currentPin' => 'required|string|size:6|regex:/^[0-9]+$/',
            'newPin' => 'required|string|size:6|regex:/^[0-9]+$/',
            'confirmPin' => 'required|same:newPin'
        ]);

        $user = auth()->user();
        
        if ($user->pin !== $this->currentPin) {
            $this->addError('currentPin', 'Current PIN is incorrect.');
            return;
        }

        $user->update([
            'pin' => $this->newPin
        ]);

        // Log the PIN change
        Activity::create([
            'log_name' => 'security',
            'description' => 'PIN changed',
            'subject_type' => get_class($user),
            'subject_id' => $user->id,
            'causer_type' => get_class($user),
            'causer_id' => $user->id,
        ]);

        $this->reset(['currentPin', 'newPin', 'confirmPin']);
        
        // Send notification
        $notificationService = app(\App\Services\NotificationService::class);
        $notificationService->notifyPinChanged($user);
        
        Flux::toast(
            variant: 'success',
            heading: 'PIN Updated!',
            text: 'Your PIN has been changed successfully.'
        );
    }

    // 2FA Management
    public function toggleTwoFactor()
    {
        if ($this->twoFactorEnabled) {
            $this->disableTwoFactor();
        } else {
            $this->showTwoFactorSetup = true;
        }
    }

    public function setupTwoFactor()
    {
        $user = auth()->user();
        
        // Get or create 2FA record
        $twoFactor = $user->twoFactorAuth;
        
        if (!$twoFactor) {
            // Create new 2FA record
            $twoFactor = new TwoFactorAuth();
            $twoFactor->user_id = $user->id;
        }
        
        // Reset verification code when setting up
        $this->twoFactorVerificationCode = '';
        
        if ($this->twoFactorMethod === 'totp') {
            $this->setupTotp($twoFactor);
        } elseif ($this->twoFactorMethod === 'email') {
            $this->setupEmail($twoFactor);
        }
    }

    public function updatedTwoFactorMethod()
    {
        // Reset setup when method changes
        $this->twoFactorSecret = '';
        $this->twoFactorQrCode = '';
        $this->twoFactorVerificationCode = '';
        $this->backupCodes = [];  // Changed to empty array
    }

    private function setupTotp($twoFactor)
    {
        // Create Google2FA instance locally when needed
        $google2fa = new Google2FA();
        $secret = $google2fa->generateSecretKey();
        
        // Save the secret to the database
        $twoFactor->secret_key = $secret;
        $twoFactor->method = 'totp';
        $twoFactor->enabled = false; // Not enabled until verified
        $twoFactor->save();
        
        $this->twoFactorSecret = $secret;
        
        // Generate QR code after setting the secret
        try {
            $this->twoFactorQrCode = $this->generateQrCode($secret);
        } catch (\Exception $e) {
            // If QR code generation fails, log it but continue
            \Log::error('QR Code generation failed: ' . $e->getMessage());
            $this->twoFactorQrCode = '';
        }
        
        $this->showTwoFactorSetup = true;
    }

    private function setupEmail($twoFactor)
    {
        $code = $twoFactor->generateEmailCode();
        
        // Send email with verification code
        $user = auth()->user();
        $expiresAt = $twoFactor->verification_code_expires_at->format('Y-m-d H:i:s T');
        
        try {
            Mail::to($user->email)->send(new TwoFactorVerificationMail($code, $user->name, $expiresAt));
            
            // Clear the verification code input for security
            $this->twoFactorVerificationCode = '';
            $this->showTwoFactorSetup = true;
            
            Flux::toast(
                variant: 'success',
                heading: 'Verification Code Sent!',
                text: 'Please check your email for the verification code and enter it below.'
            );
        } catch (\Exception $e) {
            \Log::error('Failed to send 2FA verification email: ' . $e->getMessage());
            
            // Fallback: show the code in UI if email fails
            $this->twoFactorVerificationCode = $code;
            $this->showTwoFactorSetup = true;
            
            Flux::toast(
                variant: 'warning',
                heading: 'Email Failed',
                text: 'Could not send email. Please use the code shown below.'
            );
        }
    }

    public function verifyTwoFactor()
    {
        $user = auth()->user();
        $twoFactor = $user->twoFactorAuth;
        
        if (!$twoFactor) {
            $this->addError('twoFactorVerificationCode', '2FA not set up.');
            return;
        }

        if ($this->twoFactorMethod === 'totp') {
            // Create Google2FA instance locally when needed
            $google2fa = new Google2FA();
            
            // Verify TOTP code using the saved secret from database
            if ($google2fa->verifyKey($twoFactor->secret_key, $this->twoFactorVerificationCode)) {
                $this->enableTwoFactor($twoFactor);
            } else {
                $this->addError('twoFactorVerificationCode', 'Invalid verification code.');
                return;
            }
        } elseif ($this->twoFactorMethod === 'email') {
            if ($twoFactor->verifyEmailCode($this->twoFactorVerificationCode)) {
                $this->enableTwoFactor($twoFactor);
            } else {
                $this->addError('twoFactorVerificationCode', 'Invalid verification code.');
                return;
            }
        }
    }

    private function enableTwoFactor($twoFactor)
    {
        $twoFactor->enable();
        
        // Generate backup codes
        $this->backupCodes = $twoFactor->generateBackupCodes();
        
        $this->twoFactorEnabled = true;
        $this->showTwoFactorSetup = false;
        
        // Send notification
        $notificationService = app(\App\Services\NotificationService::class);
        $notificationService->notifyTwoFactorEnabled($user, $this->twoFactorMethod);
        
        Flux::toast(
            variant: 'success',
            heading: '2FA Enabled!',
            text: 'Two-factor authentication has been enabled successfully.'
        );
    }

    /**
     * Resend email verification code for 2FA setup.
     */
    public function resendEmailCode(): void
    {
        $user = auth()->user();
        $twoFactor = $user->twoFactorAuth;
        
        if ($twoFactor && $this->twoFactorMethod === 'email') {
            $this->setupEmail($twoFactor);
            
            Flux::toast(
                variant: 'success',
                heading: 'Code Resent!',
                text: 'A new verification code has been sent to your email.'
            );
        }
    }

    private function disableTwoFactor()
    {
        $user = auth()->user();
        $twoFactor = $user->twoFactorAuth;
        
        if ($twoFactor) {
            $twoFactor->disable();
        }
        
        $this->twoFactorEnabled = false;
        $this->showTwoFactorSetup = false;
        
        // Send notification
        $notificationService = app(\App\Services\NotificationService::class);
        $notificationService->notifyTwoFactorDisabled($user);
        
        Flux::toast(
            variant: 'success',
            heading: '2FA Disabled!',
            text: 'Two-factor authentication has been disabled.'
        );
    }

    private function generateQrCode($secret)
    {
        $user = auth()->user();
        $issuer = config('app.name', 'Laravel');
        $account = $user->email;
        
        // Build the TOTP URL
        $url = sprintf(
            'otpauth://totp/%s:%s?secret=%s&issuer=%s',
            rawurlencode($issuer),
            rawurlencode($account),
            $secret,
            rawurlencode($issuer)
        );
        
        try {
            // Create QR code using Endroid library
            $qrCode = new QrCode($url);
            $qrCode->setSize(200);
            $qrCode->setMargin(1);
            
            // Create PNG writer
            $writer = new PngWriter();
            $result = $writer->write($qrCode);
            
            // Get the PNG data and return as base64 data URL
            $pngData = $result->getString();
            return 'data:image/png;base64,' . base64_encode($pngData);
        } catch (\Exception $e) {
            \Log::error('QR Code generation error: ' . $e->getMessage());
            
            // Alternative: Try with minimal options
            try {
                $qrCode = new QrCode($url);
                $qrCode->setSize(200);
                
                $writer = new PngWriter();
                $result = $writer->write($qrCode);
                
                $pngData = $result->getString();
                return 'data:image/png;base64,' . base64_encode($pngData);
            } catch (\Exception $e2) {
                \Log::error('QR Code fallback also failed: ' . $e2->getMessage());
                return '';
            }
        }
    }

    // Security Log
    public function loadSecurityLog()
    {
        $user = auth()->user();
        
        $this->securityLogs = Activity::query()
            ->causedBy($user)
            ->whereIn('log_name', ['default', 'security'])
            ->latest()
            ->take(20)
            ->get()
            ->map(function ($log) {
                return [
                    'id' => $log->id,
                    'description' => $log->description,
                    'created_at' => $log->created_at->format('Y-m-d H:i:s'), // Convert to string format
                    'created_at_human' => $log->created_at->diffForHumans(), // Add human readable format
                    'properties' => $log->properties ? $log->properties->toArray() : [] // Ensure it's an array
                ];
            })
            ->toArray();
        
        $this->showSecurityLog = true;
    }

    public function render()
    {
        return view('livewire.user.security-settings');
    }
}