<?php

namespace App\Livewire\User;

use App\Models\Country;
use App\Models\UserProfile;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use Livewire\WithFileUploads;
use Flux;

#[Layout('components.layouts.app')]
class ProfileSettings extends Component
{
    use WithFileUploads;

    #[Validate('required|string|max:255')]
    public string $first_name = '';
    
    #[Validate('nullable|string|max:255')]
    public string $middle_name = '';
    
    #[Validate('required|string|max:255')]
    public string $last_name = '';
    
    #[Validate('required|date|before:today')]
    public string $date_of_birth = '';
    
    #[Validate('required|in:male,female,other')]
    public string $gender = '';
    
    #[Validate('required|in:single,married,divorced,widowed')]
    public string $marital_status = '';
    
    #[Validate('required|string|max:255')]
    public string $occupation = '';
    
    #[Validate('nullable|numeric|min:0')]
    public ?float $annual_income = null;
    
    #[Validate('required|string|max:500')]
    public string $address_line_1 = '';
    
    #[Validate('required|string|max:255')]
    public string $city = '';
    
    #[Validate('required|string|max:255')]
    public string $state_province = '';
    
    #[Validate('required|string|max:20')]
    public string $postal_code = '';
    
    #[Validate('required|string|max:255')]
    public string $country = '';
    
    #[Validate('required|string|max:20')]
    public string $phone_number = '';
    
    #[Validate('required|string|max:255')]
    public string $emergency_contact_name = '';
    
    #[Validate('required|string|max:20')]
    public string $emergency_contact_phone = '';
    
    #[Validate('required|string|max:255')]
    public string $emergency_contact_relationship = '';
    
    #[Validate('nullable|string|max:255')]
    public string $preferred_language = 'en';
    
    #[Validate('nullable|string|max:255')]
    public string $timezone = 'UTC';
    
    #[Validate('nullable|image|max:2048')]
    public $profile_image;
    
    public ?UserProfile $userProfile = null;
    public bool $profileUpdated = false;
    public $countries = [];

    public function mount()
    {
        $this->loadUserProfile();
        $this->loadCountries();
    }

    public function loadUserProfile()
    {
        $user = Auth::user();
        $this->userProfile = $user->profile;
        
        if ($this->userProfile) {
            $this->first_name = $this->userProfile->first_name ?? '';
            $this->middle_name = $this->userProfile->middle_name ?? '';
            $this->last_name = $this->userProfile->last_name ?? '';
            $this->date_of_birth = $this->userProfile->date_of_birth?->format('Y-m-d') ?? '';
            $this->gender = $this->userProfile->gender ?? '';
            $this->marital_status = $this->userProfile->marital_status ?? '';
            $this->occupation = $this->userProfile->occupation ?? '';
            $this->annual_income = $this->userProfile->annual_income;
            $this->address_line_1 = $this->userProfile->address_line_1 ?? '';
            $this->city = $this->userProfile->city ?? '';
            $this->state_province = $this->userProfile->state_province ?? '';
            $this->postal_code = $this->userProfile->postal_code ?? '';
            $this->country = $this->userProfile->country ?? '';
            $this->phone_number = $this->userProfile->phone_number ?? '';
            $this->emergency_contact_name = $this->userProfile->emergency_contact_name ?? '';
            $this->emergency_contact_phone = $this->userProfile->emergency_contact_phone ?? '';
            $this->emergency_contact_relationship = $this->userProfile->emergency_contact_relationship ?? '';
            $this->preferred_language = $this->userProfile->preferred_language ?? 'en';
            $this->timezone = $this->userProfile->timezone ?? 'UTC';
        }
    }

    public function loadCountries()
    {
        $this->countries = Country::orderBy('name')->get();
    }

    public function updateProfile()
    {
        $this->validate();

        $user = Auth::user();
        
        if (!$this->userProfile) {
            $this->userProfile = new UserProfile();
            $this->userProfile->user_id = $user->id;
        }

        // Handle profile image upload
        if ($this->profile_image) {
            $this->handleProfileImageUpload();
        }

        // Update profile fields
        $this->userProfile->fill([
            'first_name' => $this->first_name,
            'middle_name' => $this->middle_name,
            'last_name' => $this->last_name,
            'date_of_birth' => $this->date_of_birth,
            'gender' => $this->gender,
            'marital_status' => $this->marital_status,
            'occupation' => $this->occupation,
            'annual_income' => $this->annual_income ? (float) $this->annual_income : null,
            'address_line_1' => $this->address_line_1,
            'city' => $this->city,
            'state_province' => $this->state_province,
            'postal_code' => $this->postal_code,
            'country' => $this->country,
            'phone_number' => $this->phone_number,
            'emergency_contact_name' => $this->emergency_contact_name,
            'emergency_contact_phone' => $this->emergency_contact_phone,
            'emergency_contact_relationship' => $this->emergency_contact_relationship,
            'preferred_language' => $this->preferred_language,
            'timezone' => $this->timezone,
        ]);

        $this->userProfile->save();

        $this->profileUpdated = true;

        Flux::toast(
            variant: 'success',
            heading: 'Profile Updated Successfully!',
            text: 'Your profile information has been saved.'
        );
    }

private function handleProfileImageUpload()
{
    // Delete old profile image if it exists
    if ($this->userProfile && $this->userProfile->profile_image) {
        $oldPath = public_path($this->userProfile->profile_image);
        if (file_exists($oldPath)) {
            @unlink($oldPath);
        }
    }

    // Store new profile image directly in /public/profile-images
    $file = $this->profile_image;
    if ($file && $file->isValid()) {
        $folder = 'profile-images';
        $extension = $file->getClientOriginalExtension();
        $filename = 'profile-' . time() . '.' . $extension;
        $destination = public_path($folder);

        // Create folder if not exists
        if (!file_exists($destination)) {
            mkdir($destination, 0777, true);
        }

        // Safely copy file instead of moving (avoids permission issues)
        $targetPath = $destination . '/' . $filename;
        file_put_contents($targetPath, file_get_contents($file->getRealPath()));

        // Save relative path (e.g., profile-images/profile-1234.png)
        $this->userProfile->profile_image = $folder . '/' . $filename;
        $this->userProfile->profile_image_disk = 'public';
    }

    // Clear temporary upload
    $this->profile_image = null;
}

public function removeProfileImage()
{
    if ($this->userProfile && $this->userProfile->profile_image) {
        $filePath = public_path($this->userProfile->profile_image);

        if (file_exists($filePath)) {
            @unlink($filePath);
        }

        $this->userProfile->profile_image = null;
        $this->userProfile->profile_image_disk = null;
        $this->userProfile->save();

        Flux::toast(
            variant: 'success',
            heading: 'Profile Image Removed!',
            text: 'Your profile image has been removed.'
        );
    }
}

    public function render()
    {
        return view('livewire.user.profile-settings');
    }
}
