<?php

namespace App\Livewire\User;

use App\Models\Account;
use App\Models\Transaction;
use App\Services\ExchangeRateService;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Rule;
use Flux;

#[Layout('components.layouts.app')]
class InterAccountTransfer extends Component
{
    #[Rule('required|exists:accounts,id')]
    public $fromAccountId;

    #[Rule('required|exists:accounts,id')]
    public $toAccountId;

    #[Rule('required|numeric|min:0.01')]
    public $amount;

    #[Rule('nullable|string|max:255')]
    public $description = '';

    public $exchangeRate = 1.0;
    public $convertedAmount = 0;
    public $showExchangeRate = false;
    public $loading = false;
    public $error = '';

    public $userAccounts = [];
    public $fromAccount = null;
    public $toAccount = null;

    protected $exchangeRateService;

    public function boot(ExchangeRateService $exchangeRateService)
    {
        $this->exchangeRateService = $exchangeRateService;
    }

    public function mount()
    {
        $this->loadUserAccounts();
    }

    public function loadUserAccounts()
    {
        $this->userAccounts = auth()->user()->accounts()
            ->where('status', 'active')
            ->get(['id', 'account_number', 'account_type', 'currency', 'balance', 'is_master_account'])
            ->map(function ($account) {
                return [
                    'id' => $account->id,
                    'label' => "{$account->account_type} ({$account->currency}) - {$account->account_number}",
                    'currency' => $account->currency,
                    'balance' => $account->balance,
                    'is_master_account' => $account->is_master_account,
                ];
            });
    }

    public function updatedFromAccountId()
    {
        $this->fromAccount = $this->userAccounts->firstWhere('id', $this->fromAccountId);
        $this->calculateExchangeRate();
    }

    public function updatedToAccountId()
    {
        $this->toAccount = $this->userAccounts->firstWhere('id', $this->toAccountId);
        $this->calculateExchangeRate();
    }

    public function updatedAmount()
    {
        $this->calculateExchangeRate();
    }

    public function calculateExchangeRate()
    {
        if (!$this->fromAccount || !$this->toAccount || !$this->amount) {
            $this->showExchangeRate = false;
            $this->convertedAmount = 0;
            return;
        }

        if ($this->fromAccount['currency'] === $this->toAccount['currency']) {
            $this->exchangeRate = 1.0;
            $this->convertedAmount = $this->amount;
            $this->showExchangeRate = false;
            return;
        }

        $this->loading = true;
        $this->showExchangeRate = true;

        try {
            $result = $this->exchangeRateService->convertCurrency(
                $this->amount,
                $this->fromAccount['currency'],
                $this->toAccount['currency']
            );

            if ($result['success']) {
                $this->exchangeRate = $result['exchange_rate'];
                $this->convertedAmount = $result['converted_amount'];
            } else {
                $this->error = $result['error'];
                $this->showExchangeRate = false;
            }
        } catch (\Exception $e) {
            $this->error = 'Failed to calculate exchange rate. Please try again.';
            $this->showExchangeRate = false;
        } finally {
            $this->loading = false;
        }
    }

    public function transfer()
    {
        $this->validate();

        if ($this->fromAccountId === $this->toAccountId) {
            $this->error = 'Cannot transfer to the same account.';
            return;
        }

        if (!$this->fromAccount || !$this->toAccount) {
            $this->error = 'Please select valid accounts.';
            return;
        }

        if ($this->amount > $this->fromAccount['balance']) {
            $this->error = 'Insufficient balance in the source account.';
            return;
        }

        $this->loading = true;
        $this->error = '';

        try {
            DB::transaction(function () {
                // Deduct from source account
                $fromAccount = Account::find($this->fromAccountId);
                $fromAccount->decrement('balance', $this->amount);
                $fromAccount->decrement('available_balance', $this->amount);

                // Add to destination account (with converted amount if different currency)
                $toAccount = Account::find($this->toAccountId);
                $amountToAdd = $this->fromAccount['currency'] === $this->toAccount['currency'] 
                    ? $this->amount 
                    : $this->convertedAmount;
                
                $toAccount->increment('balance', $amountToAdd);
                $toAccount->increment('available_balance', $amountToAdd);

                // Create transaction records using correct field names
                Transaction::create([
                    'transaction_id' => 'TRF-' . strtoupper(uniqid()),
                    'user_id' => auth()->id(),
                    'from_account_id' => $this->fromAccountId,
                    'to_account_id' => null,
                    'type' => 'exchange',
                    'category' => 'exchange',
                    'subcategory' => 'inter_account',
                    'amount' => $this->amount,
                    'currency' => $this->fromAccount['currency'],
                    'exchange_rate' => $this->exchangeRate,
                    'converted_amount' => $this->convertedAmount,
                    'description' => $this->description,
                    'status' => 'completed',
                    'processed_at' => now(),
                    'completed_at' => now(),
                    'metadata' => [
                        'transfer_type' => 'inter_account',
                        'from_currency' => $this->fromAccount['currency'],
                        'to_currency' => $this->toAccount['currency'],
                        'original_amount' => $this->amount,
                        'converted_amount' => $this->convertedAmount,
                    ]
                ]);

                // Create reverse transaction for destination account
                Transaction::create([
                    'transaction_id' => 'TRF-' . strtoupper(uniqid()),
                    'user_id' => auth()->id(),
                    'from_account_id' => null,
                    'to_account_id' => $this->toAccountId,
                    'type' => 'exchange',
                    'category' => 'exchange',
                    'subcategory' => 'inter_account',
                    'amount' => $amountToAdd,
                    'currency' => $this->toAccount['currency'],
                    'exchange_rate' => 1 / $this->exchangeRate,
                    'converted_amount' => $this->amount,
                    'description' => "Received: {$this->description}",
                    'status' => 'completed',
                    'processed_at' => now(),
                    'completed_at' => now(),
                    'metadata' => [
                        'transfer_type' => 'inter_account',
                        'from_currency' => null,
                        'to_currency' => $this->toAccount['currency'],
                        'original_amount' => $this->amount,
                        'converted_amount' => $this->convertedAmount,
                    ]
                ]);
            });

            // Store values before resetting
            $transferAmount = $this->amount;
            $fromCurrency = $this->fromAccount['currency'];
            $fromAccountLabel = $this->fromAccount['label'];
            $toAccountLabel = $this->toAccount['label'];

            $this->reset(['amount', 'description']);
            $this->loadUserAccounts();

            // Send notification
            $notificationService = app(\App\Services\NotificationService::class);
            $notificationService->notifyInterAccountTransfer(
                auth()->user(),
                $transferAmount,
                $fromCurrency,
                $fromAccountLabel,
                $toAccountLabel
            );

            Flux::toast(
                variant: 'success',
                heading: 'Transfer Completed!',
                text: 'Your funds have been transferred successfully.'
            );

            sleep(0.5);
            $this->dispatch('$refresh');

        } catch (\Exception $e) {
            $this->error = 'Transfer failed. Please try again.';
            Flux::toast(
                variant: 'danger',
                heading: 'Transfer Failed!',
                text: 'Your funds have not been transferred.'
            );
            logger()->error('Transfer error', [
                'user_id' => auth()->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        } finally {
            $this->loading = false;
        }
    }

    public function resetForm()
    {
        $this->reset(['fromAccountId', 'toAccountId', 'amount', 'description']);
        $this->showExchangeRate = false;
        $this->convertedAmount = 0;
    }

    public function render()
    {
        return view('livewire.user.inter-account-transfer');
    }
}
