<?php

namespace App\Livewire\User\Banking;

use App\Models\KycDocument;
use App\Models\UserProfile;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Attributes\Layout;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Flux;

#[Layout('components.layouts.app')]
class KycVerification extends Component
{
    use WithFileUploads;

    // Document upload properties
    public $documents = [];
    public $documentType = '';
    public $documentName = '';
    public $uploadedFiles = [];
    
    // KYC status properties
    public $kycStatus = 'pending';
    public $kycProgress = 0;
    public $requiredDocuments = [];
    public $uploadedDocuments = [];
    public $approvedDocuments = [];
    
    // Form state
    public $showUploadForm = false;
    public $selectedDocumentType = '';
    public $isUploading = false;

    protected $rules = [
        'documentType' => 'required|string|in:id,passport,utility_bill,bank_statement,proof_of_address',
        'documentName' => 'required|string|max:255',
        'documents.*' => 'required|file|mimes:pdf,jpg,jpeg,png|max:10240', // 10MB max
    ];

    protected $messages = [
        'documents.*.required' => 'Please select a document to upload.',
        'documents.*.mimes' => 'Only PDF, JPG, JPEG, and PNG files are allowed.',
        'documents.*.max' => 'File size must not exceed 10MB.',
        'documentType.required' => 'Please select a document type.',
        'documentName.required' => 'Please enter a document name.',
    ];

    public function mount()
    {
        $this->loadKycStatus();
        $this->loadRequiredDocuments();
        $this->loadUploadedDocuments();
        $this->clearKycSession();
        $this->checkKycRequirement();
    }
    
    public function checkKycRequirement()
    {
        $isKycRequired = \App\Helpers\SettingsHelper::isKycRequired();
        
        if ($isKycRequired) {
            $user = auth()->user();
            $profile = $user->profile;
            
            if (!$profile || $profile->kyc_status !== 'verified') {
                if (!$user->hasRole(['super-admin', 'admin'])) {
                    $this->js("
                        setTimeout(() => {
                            if (window.Flux && window.Flux.toast) {
                                window.Flux.toast({
                                    variant: 'warning',
                                    heading: 'KYC Verification Required',
                                    text: 'Please complete your identity verification to access banking features.',
                                    duration: 6000
                                });
                            }
                        }, 100);
                    ");
                }
            }
        }
    }
    
    public function clearKycSession()
    {
        if (session()->has('kyc_required')) {
            session()->forget(['kyc_required', 'kyc_variant', 'kyc_heading', 'kyc_text']);
        }
    }

    public function loadKycStatus()
    {
        $user = auth()->user();
        
        if (!$user) {
            $this->kycStatus = 'not_started';
            return;
        }
        
        $profile = $user->profile;
        
        if ($profile) {
            $this->kycStatus = $profile->kyc_status ?? 'pending';
        } else {
            $this->kycStatus = 'not_started';
        }
        
        $this->calculateKycProgress();
    }

    public function loadRequiredDocuments()
    {
        $this->requiredDocuments = [
            'id' => [
                'name' => 'Government ID',
                'description' => 'Driver\'s license, national ID, or state ID',
                'required' => true,
                'uploaded' => false,
                'approved' => false,
            ],
            'passport' => [
                'name' => 'Passport',
                'description' => 'Valid passport (alternative to government ID)',
                'required' => false,
                'uploaded' => false,
                'approved' => false,
            ],
            'utility_bill' => [
                'name' => 'Utility Bill',
                'description' => 'Recent utility bill (electricity, water, gas)',
                'required' => true,
                'uploaded' => false,
                'approved' => false,
            ],
            'proof_of_address' => [
                'name' => 'Proof of Address',
                'description' => 'Bank statement, lease agreement, or official mail',
                'required' => true,
                'uploaded' => false,
                'approved' => false,
            ],
        ];
    }

    public function loadUploadedDocuments()
    {
        $user = auth()->user();
        
        if (!$user) {
            $this->uploadedDocuments = [];
            return;
        }
        
        $documents = KycDocument::where('user_id', $user->id)->get();
        
        $this->uploadedDocuments = $documents->map(function ($document) {
            return [
                'id' => $document->id,
                'document_name' => $document->document_name,
                'document_type' => $document->document_type,
                'document_type_label' => $this->getDocumentTypeLabel($document->document_type),
                'file_name' => $document->file_name,
                'file_size' => $document->file_size,
                'file_size_human' => $this->formatFileSize($document->file_size),
                'status' => $document->status,
                'status_label' => $this->getStatusLabel($document->status),
                'created_at' => $document->created_at,
                'reviewed_at' => $document->reviewed_at,
                'rejection_reason' => $document->rejection_reason,
            ];
        })->toArray();
        
        // Update required documents status
        foreach ($documents as $document) {
            if (isset($this->requiredDocuments[$document->document_type])) {
                $this->requiredDocuments[$document->document_type]['uploaded'] = true;
                $this->requiredDocuments[$document->document_type]['approved'] = $document->status === 'approved';
            }
        }
        
        $this->calculateKycProgress();
    }

    public function calculateKycProgress()
    {
        $totalRequired = collect($this->requiredDocuments)->where('required', true)->count();
        $approvedRequired = collect($this->requiredDocuments)
            ->where('required', true)
            ->where('approved', true)
            ->count();
        
        $this->kycProgress = $totalRequired > 0 ? round(($approvedRequired / $totalRequired) * 100) : 0;
    }

    public function openUploadModal($documentType = null)
    {
        $this->selectedDocumentType = $documentType ?? '';
        $this->documentType = $documentType ?? '';
        $this->documentName = '';
        $this->documents = [];
        $this->showUploadForm = true;
        $this->resetValidation();
    }

    public function hideUploadForm()
    {
        $this->showUploadForm = false;
        $this->selectedDocumentType = '';
        $this->documentType = '';
        $this->documentName = '';
        $this->documents = [];
        $this->resetValidation();
    }

    public function updatedDocuments()
    {
        $this->validate([
            'documents.*' => 'required|file|mimes:pdf,jpg,jpeg,png|max:10240',
        ]);
    }

    public function uploadDocuments()
    {
        $this->validate();
        
        if (empty($this->documents)) {
            $this->addError('documents', 'Please select at least one document to upload.');
            return;
        }

        $this->isUploading = true;
        
        try {
            $user = auth()->user();
            $uploadedCount = 0;
            
            foreach ($this->documents as $document) {
                // Generate unique filename
                $extension = $document->getClientOriginalExtension();
                $filename = Str::uuid() . '.' . $extension;
                $path = 'kyc-documents/' . $user->id . '/' . $filename;
                
               $destinationPath = public_path('kyc-documents/' . $user->id);

// Ensure directory exists
if (!file_exists($destinationPath)) {
    mkdir($destinationPath, 0755, true);
}

// Generate a new filename
$filename = Str::uuid() . '.' . $document->getClientOriginalExtension();

// Move using PHP directly (not Livewire’s move)
copy($document->getRealPath(), $destinationPath . '/' . $filename);

// Store relative path manually
$storedPath = 'kyc-documents/' . $user->id . '/' . $filename;

                
                // Create database record
                KycDocument::create([
                    'user_id' => $user->id,
                    'document_type' => $this->documentType,
                    'document_name' => $this->documentName,
                    'file_path' => $storedPath,
                    'file_name' => $document->getClientOriginalName(),
                    'mime_type' => $document->getMimeType(),
                    'file_size' => $document->getSize(),
                    'status' => 'pending',
                    'metadata' => [
                        'original_name' => $document->getClientOriginalName(),
                        'uploaded_at' => now()->toISOString(),
                        'ip_address' => request()->ip(),
                        'user_agent' => request()->userAgent(),
                    ],
                ]);
                
                $uploadedCount++;
            }
            
            // Update user profile KYC status if needed
            $profile = $user->profile;
            if ($profile && $profile->kyc_status === 'not_started') {
                $profile->update(['kyc_status' => 'pending']);
            }
            
            // Reload data
            $this->loadUploadedDocuments();
            $this->hideUploadForm();
            
            Flux::toast(
                variant: 'success',
                heading: 'Documents Uploaded Successfully!',
                text: "{$uploadedCount} document(s) uploaded and submitted for review."
            );
            
        } catch (\Exception $e) {
            Flux::toast(
                variant: 'danger',
                heading: 'Upload Failed',
                text: 'An error occurred while uploading your documents. Please try again.'
            );
            
            \Log::error('KYC Document Upload Error: ' . $e->getMessage());
        } finally {
            $this->isUploading = false;
        }
    }

    public function deleteDocument($documentId)
    {
        try {
            $document = KycDocument::where('user_id', auth()->id())
                ->where('id', $documentId)
                ->first();
            
            if ($document && $document->status === 'pending') {
                // Delete file from storage
                if (Storage::disk('public')->exists($document->file_path)) {
                    Storage::disk('public')->delete($document->file_path);
                }
                
                // Delete database record
                $document->delete();
                
                // Reload data
                $this->loadUploadedDocuments();
                
                Flux::toast(
                    variant: 'success',
                    heading: 'Document Deleted',
                    text: 'Document has been successfully deleted.'
                );
            } else {
                Flux::toast(
                    variant: 'danger',
                    heading: 'Cannot Delete',
                    text: 'Only pending documents can be deleted.'
                );
            }
        } catch (\Exception $e) {
            Flux::toast(
                variant: 'danger',
                heading: 'Delete Failed',
                text: 'An error occurred while deleting the document.'
            );
        }
    }

    public function getDocumentStatusColor($status)
    {
        return match ($status) {
            'approved' => 'bg-green-100 text-green-800',
            'rejected' => 'bg-red-100 text-red-800',
            'expired' => 'bg-gray-100 text-gray-800',
            default => 'bg-amber-100 text-amber-800',
        };
    }

    public function getDocumentStatusIcon($status)
    {
        return match ($status) {
            'approved' => 'check-circle',
            'rejected' => 'x-circle',
            'expired' => 'clock',
            default => 'clock',
        };
    }

    public function getKycStatusColor()
    {
        return match ($this->kycStatus) {
            'verified' => 'bg-green-100 text-green-800',
            'rejected' => 'bg-red-100 text-red-800',
            'pending' => 'bg-amber-100 text-amber-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    public function getKycStatusLabel()
    {
        return match ($this->kycStatus) {
            'verified' => 'Verified',
            'rejected' => 'Rejected',
            'pending' => 'Pending Review',
            default => 'Not Started',
        };
    }

    public function getDocumentTypeLabel($type)
    {
        return match ($type) {
            'id' => 'Government ID',
            'passport' => 'Passport',
            'utility_bill' => 'Utility Bill',
            'bank_statement' => 'Bank Statement',
            'proof_of_address' => 'Proof of Address',
            default => ucfirst(str_replace('_', ' ', $type)),
        };
    }

    public function getStatusLabel($status)
    {
        return match ($status) {
            'approved' => 'Approved',
            'rejected' => 'Rejected',
            'expired' => 'Expired',
            default => 'Pending Review',
        };
    }

    public function formatFileSize($bytes)
    {
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } else {
            return $bytes . ' bytes';
        }
    }

    public function render()
    {
        return view('livewire.user.banking.kyc-verification');
    }
}
