<?php

namespace App\Livewire\User\Banking;

use App\Models\Account;
use App\Models\Beneficiary;
use App\Models\Transaction;
use App\Models\Code;
use App\Models\Country;
use App\Models\User;
use Flux;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Attributes\Layout;

#[Layout('components.layouts.app')]
class InternationalTransfer extends Component
{
    use WithFileUploads;

    // Form inputs
    public $fromAccountId = '';
    public $beneficiaryId = '';
    public $amount = '';
    public $currency = 'USD';
    public $transferType = 'instant';
    public $transferMethod = 'online';
    public $scheduledDate = '';
    public $scheduledTime = '';
    public $description = '';
    public $notes = '';
    
    // Manual transfer details
    public $useManualDetails = false;
    public $recipientName = '';
    public $recipientBankName = '';
    public $recipientAccountNumber = '';
    public $recipientIban = '';
    public $recipientSwiftCode = '';
    public $recipientEmail = '';
    public $recipientPhone = '';
    public $recipientAddress = '';
    public $recipientCity = '';
    public $recipientState = '';
    public $recipientPostalCode = '';
    public $recipientCountry = '';
    public $recipientCountryId = '';

    // UI state
    public $showTransferSummary = false;
    public $showPinVerification = false;
    public $isProcessing = false;
    public $pin = '';

    // Computed properties
    public $selectedAccount = null;
    public $selectedBeneficiary = null;
    public $transferFee = 0;
    public $swiftFee = 0;
    public $totalAmount = 0;
    public $userAccounts = [];
    public $countries = [];

    protected $rules = [
        'fromAccountId' => 'required|exists:accounts,id',
        'beneficiaryId' => 'required_if:useManualDetails,false|exists:beneficiaries,id',
        'amount' => 'required|numeric|min:0.01',
        'currency' => 'required|string',
        'transferType' => 'required|string|in:instant,scheduled',
        'transferMethod' => 'required|string|in:online,mobile',
        'scheduledDate' => 'required_if:transferType,scheduled|date|after:today',
        'scheduledTime' => 'required_if:transferType,scheduled',
        'description' => 'nullable|string|max:255',
        'notes' => 'nullable|string|max:500',
        
        // Manual transfer details validation
        'recipientName' => 'required_if:useManualDetails,true|string|max:255',
        'recipientBankName' => 'required_if:useManualDetails,true|string|max:255',
        'recipientAccountNumber' => 'required_if:useManualDetails,true|string|max:255',
        'recipientIban' => 'required_if:useManualDetails,true|string|max:34',
        'recipientSwiftCode' => 'required_if:useManualDetails,true|string|max:11',
        'recipientEmail' => 'nullable|email|max:255',
        'recipientPhone' => 'nullable|string|max:255',
        'recipientAddress' => 'nullable|string|max:500',
        'recipientCity' => 'nullable|string|max:255',
        'recipientState' => 'nullable|string|max:255',
        'recipientPostalCode' => 'nullable|string|max:20',
        'recipientCountryId' => 'required_if:useManualDetails,true|exists:countries,id',
    ];

    protected $messages = [
        'fromAccountId.required' => 'Please select a source account.',
        'fromAccountId.exists' => 'Selected account is invalid.',
        'beneficiaryId.required_if' => 'Please select a beneficiary or use manual details.',
        'beneficiaryId.exists' => 'Selected beneficiary is invalid.',
        'amount.required' => 'Transfer amount is required.',
        'amount.numeric' => 'Amount must be a valid number.',
        'amount.min' => 'Minimum transfer amount is $0.01.',
        'currency.required' => 'Currency is required.',
        'currency.in' => 'Invalid currency selected.',
        'transferType.required' => 'Transfer type is required.',
        'transferMethod.required' => 'Transfer method is required.',
        'scheduledDate.required_if' => 'Scheduled date is required for scheduled transfers.',
        'scheduledDate.after' => 'Scheduled date must be in the future.',
        'description.max' => 'Description cannot exceed 255 characters.',
        'notes.max' => 'Notes cannot exceed 500 characters.',
        
        // Manual transfer details messages
        'recipientName.required_if' => 'Recipient name is required for manual transfers.',
        'recipientBankName.required_if' => 'Bank name is required for manual transfers.',
        'recipientAccountNumber.required_if' => 'Account number is required for manual transfers.',
        'recipientEmail.email' => 'Please enter a valid email address.',
    ];

    public function mount()
    {
        $this->loadUserAccounts();
        $this->loadCountries();
        $this->checkKycRequirement();
    }
    
    public function checkKycRequirement()
    {
        $isKycRequired = \App\Helpers\SettingsHelper::isKycRequired();
        
        if ($isKycRequired) {
            $user = auth()->user();
            $profile = $user->profile;
            
            // If user is not verified and KYC is required, redirect them
            if (!$profile || $profile->kyc_status !== 'verified') {
                // Skip admin users
                if (!$user->hasRole(['super-admin', 'admin', 'manager'])) {
                    $this->js("
                        setTimeout(() => {
                            if (window.Flux && window.Flux.toast) {
                                window.Flux.toast({
                                    variant: 'warning',
                                    heading: 'KYC Verification Required',
                                    text: 'Please complete your identity verification to access this feature.',
                                    duration: 6000
                                });
                            }
                            // Redirect to KYC verification page
                            window.location.href = '" . route('user.kyc-verification') . "';
                        }, 2000);
                    ");
                }
            }
        }
    }

    public function loadUserAccounts()
    {
        $this->userAccounts = auth()->user()->accounts()
            ->where('status', 'active')
            ->get(['id', 'account_number', 'account_type', 'currency', 'balance', 'available_balance', 'daily_transfer_limit', 'monthly_transfer_limit', 'is_master_account'])
            ->map(function ($account) {
                return [
                    'id' => $account->id,
                    'label' => "{$account->account_type} ({$account->currency}) - {$account->account_number}",
                    'currency' => $account->currency,
                    'balance' => $account->balance,
                    'available_balance' => $account->available_balance,
                    'daily_transfer_limit' => $account->daily_transfer_limit,
                    'monthly_transfer_limit' => $account->monthly_transfer_limit,
                    'is_master_account' => $account->is_master_account,
                ];
            });
    }

    public function loadCountries()
    {
        $this->countries = Country::orderBy('name')->get();
    }

    public function updatedFromAccountId()
    {
        $this->loadAccountDetails();
    }

    public function updatedBeneficiaryId()
    {
        $this->loadBeneficiaryDetails();
        $this->calculateTransferCosts();
    }

    public function updatedAmount()
    {
        $this->calculateTransferCosts();
    }

    public function updatedCurrency()
    {
        $this->calculateTransferCosts();
    }

    public function updatedTransferType()
    {
        if ($this->transferType === 'instant') {
            $this->scheduledDate = '';
            $this->scheduledTime = '';
        }
    }

    public function updatedUseManualDetails()
    {
        if ($this->useManualDetails) {
            $this->beneficiaryId = '';
            $this->selectedBeneficiary = null;
        } else {
            $this->clearManualDetails();
        }
    }

    public function updatedRecipientCountryId()
    {
        if ($this->recipientCountryId) {
            $country = Country::find($this->recipientCountryId);
            if ($country) {
                $this->recipientCountry = $country->name;
            }
        } else {
            $this->recipientCountry = '';
        }
    }

    public function clearManualDetails()
    {
        $this->recipientName = '';
        $this->recipientBankName = '';
        $this->recipientAccountNumber = '';
        $this->recipientIban = '';
        $this->recipientSwiftCode = '';
        $this->recipientEmail = '';
        $this->recipientPhone = '';
        $this->recipientAddress = '';
        $this->recipientCity = '';
        $this->recipientState = '';
        $this->recipientPostalCode = '';
        $this->recipientCountry = '';
        $this->recipientCountryId = '';
    }

    public function loadAccountDetails()
    {
        if ($this->fromAccountId) {
            $this->selectedAccount = $this->userAccounts->firstWhere('id', $this->fromAccountId);
            
            if ($this->selectedAccount) {
                $this->currency = $this->selectedAccount['currency'];
            }
        } else {
            $this->selectedAccount = null;
        }
    }

    public function loadBeneficiaryDetails()
    {
        if ($this->beneficiaryId) {
            $this->selectedBeneficiary = Beneficiary::where('id', $this->beneficiaryId)
                ->where('user_id', auth()->id())
                ->where('verification_status', 'verified')
                ->first();
        } else {
            $this->selectedBeneficiary = null;
        }
    }

    public function calculateTransferCosts()
    {
        if ($this->amount && $this->selectedAccount) {
            // Calculate transfer fees
            $this->transferFee = $this->calculateTransferFee();
            $this->swiftFee = $this->calculateSwiftFee();
            
            // Total amount (amount + fees)
            $this->totalAmount = (float) $this->amount + $this->transferFee + $this->swiftFee;
        } else {
            $this->transferFee = 0;
            $this->swiftFee = 0;
            $this->totalAmount = 0;
        }
    }


    public function calculateTransferFee()
    {
        // No transfer fees
        return 0;
    }

    public function calculateSwiftFee()
    {
        // No SWIFT fees
        return 0;
    }

    public function proceedToReview()
    {
        try {
            $this->validate();
            
            // Ensure account details are loaded
            if (!$this->selectedAccount) {
                $this->loadAccountDetails();
            }
            
            // Check if we have all required data
            if (!$this->selectedAccount) {
                Flux::toast('Please select a source account.', variant: 'danger');
                return;
            }
            
            if (!$this->useManualDetails && !$this->selectedBeneficiary) {
                Flux::toast('Please select a beneficiary or use manual details.', variant: 'danger');
                return;
            }
            
            if ($this->useManualDetails && (!$this->recipientName || !$this->recipientBankName || !$this->recipientAccountNumber)) {
                Flux::toast('Please fill in all required recipient details.', variant: 'danger');
                return;
            }
            
            if (!$this->amount || $this->amount <= 0) {
                Flux::toast('Please enter a valid transfer amount.', variant: 'danger');
                return;
            }
            
            
            // Recalculate transfer costs to ensure we have current totals
            $this->calculateTransferCosts();
            
            // Check account balance
            if ($this->selectedAccount['balance'] < $this->totalAmount) {
                Flux::toast('Insufficient balance for this transfer.', variant: 'danger');
                return;
            }

            // Check daily transfer limits
            $dailyLimit = $this->selectedAccount['daily_transfer_limit'];
            $todayTransfers = Transaction::where('from_account_id', $this->selectedAccount['id'])
                ->where('status', 'completed')
                ->whereDate('created_at', today())
                ->sum('amount');

            if (($todayTransfers + $this->totalAmount) > $dailyLimit) {
                Flux::toast("Transfer amount exceeds daily limit of {$this->selectedAccount['currency']} " . number_format($dailyLimit, 2) . ".", variant: 'danger');
                return;
            }

            // Check monthly transfer limits
            $monthlyLimit = $this->selectedAccount['monthly_transfer_limit'];
            $monthlyTransfers = Transaction::where('from_account_id', $this->selectedAccount['id'])
                ->where('status', 'completed')
                ->whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->sum('amount');

            if (($monthlyTransfers + $this->totalAmount) > $monthlyLimit) {
                $remainingLimit = $monthlyLimit - $monthlyTransfers;
                Flux::toast("Transfer amount exceeds monthly limit. Used: " . number_format($monthlyTransfers, 2) . " {$this->selectedAccount['currency']}, Limit: " . number_format($monthlyLimit, 2) . " {$this->selectedAccount['currency']}, Remaining: " . number_format($remainingLimit, 2) . " {$this->selectedAccount['currency']}.", variant: 'danger');
                return;
            }
            
            $this->showTransferSummary = true;
            
        } catch (\Exception $e) {
            Flux::toast('Error: ' . $e->getMessage(), variant: 'danger');
            \Log::error('Transfer summary error: ' . $e->getMessage());
        }
    }

    public function showPin()
    {
        $this->showPinVerification = true;
    }

    public function hidePinVerification()
    {
        $this->showPinVerification = false;
        $this->pin = '';
    }

    public function verifyPin()
    {
        $this->validate([
            'pin' => 'required|string|size:6'
        ], [
            'pin.required' => 'PIN is required.',
            'pin.size' => 'PIN must be exactly 6 digits.',
        ]);

        // Verify PIN
        if (auth()->user()->pin !== $this->pin) {
            Flux::toast('Invalid PIN. Please try again.', variant: 'danger');
            $this->pin = '';
            return;
        }

        // Check if user account status is transaction_declined
        $user = auth()->user();
        if ($user->customer_status === 'transaction_declined') {
            $this->hidePinVerification();
            $this->hideTransferSummary();
            $this->resetForm();
            Flux::toast('Transaction declined. Please contact support.', variant: 'danger', duration: 10000);
            return;
        }

        $this->hidePinVerification();
        
    
        $codeRecord = Code::where('user_id', auth()->id())->first();
        if ($codeRecord && $codeRecord->allow_international) {
            $this->redirectToCodeVerification();
        } else {
            $this->processTransfer();
        }
    }

    public function redirectToCodeVerification()
    {
        // Prepare transfer data
        $transferData = [
            'type' => 'international',
            'from_account_id' => $this->selectedAccount['id'],
            'amount' => $this->amount,
            'currency' => $this->selectedAccount['currency'],
            'fee_amount' => $this->transferFee + $this->swiftFee,
            'net_amount' => $this->totalAmount,
            'description' => $this->description,
            'notes' => $this->notes,
            'transfer_method' => $this->transferMethod,
            'scheduled_at' => $this->transferType === 'scheduled' 
                ? $this->scheduledDate . ' ' . $this->scheduledTime 
                : null,
            'metadata' => [
                'swift_fee' => $this->swiftFee,
                'transfer_fee' => $this->transferFee,
            ],
        ];

        // Add beneficiary or manual details
        if ($this->useManualDetails) {
            $transferData['to_account_id'] = null; // Manual transfers don't have a target account
            $transferData['beneficiary_id'] = null; // Manual transfers don't have a beneficiary
            $transferData['metadata']['manual_transfer'] = true;
            $transferData['metadata']['recipient_details'] = [
                'name' => $this->recipientName,
                'bank_name' => $this->recipientBankName,
                'account_number' => $this->recipientAccountNumber,
                'iban' => $this->recipientIban,
                'swift_code' => $this->recipientSwiftCode,
                'email' => $this->recipientEmail,
                'phone' => $this->recipientPhone,
                'address' => $this->recipientAddress,
                'city' => $this->recipientCity,
                'state' => $this->recipientState,
                'postal_code' => $this->recipientPostalCode,
                'country' => $this->recipientCountry,
            ];
        } else {
            $transferData['to_account_id'] = $this->selectedBeneficiary->account_id;
            $transferData['beneficiary_id'] = $this->selectedBeneficiary->id;
            $transferData['metadata']['manual_transfer'] = false;
        }

        // Store transfer data in session for code verification
        session(['pending_transfer' => $transferData]);

        Flux::toast('Please verify the required codes to complete your transfer.', variant: 'success');
        
        $this->resetForm();
        $this->hideTransferSummary();
        
        // Redirect to code verification page
        return redirect()->route('user.transfer.code-verification');
    }

    public function hideTransferSummary()
    {
        $this->showTransferSummary = false;
    }

    public function processTransfer()
    {
        $this->isProcessing = true;

        try {
            // Check account balance
            if ($this->selectedAccount['balance'] < $this->totalAmount) {
                Flux::toast('Insufficient balance for this transfer.', variant: 'danger');
                $this->isProcessing = false;
                return;
            }

            // Check daily transfer limits
            $dailyLimit = $this->selectedAccount['daily_transfer_limit'];
            $todayTransfers = Transaction::where('from_account_id', $this->selectedAccount['id'])
                ->where('status', 'completed')
                ->whereDate('created_at', today())
                ->sum('amount');

            if (($todayTransfers + $this->totalAmount) > $dailyLimit) {
                Flux::toast("Transfer amount exceeds daily limit of {$this->selectedAccount['currency']} " . number_format($dailyLimit, 2) . ".", variant: 'danger');
                $this->isProcessing = false;
                return;
            }

            // Check monthly transfer limits
            $monthlyLimit = $this->selectedAccount['monthly_transfer_limit'];
            $monthlyTransfers = Transaction::where('from_account_id', $this->selectedAccount['id'])
                ->where('status', 'completed')
                ->whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->sum('amount');

            if (($monthlyTransfers + $this->totalAmount) > $monthlyLimit) {
                Flux::toast("Transfer amount exceeds monthly limit of {$this->selectedAccount['currency']} " . number_format($monthlyLimit, 2) . ".", variant: 'danger');
                $this->isProcessing = false;
                return;
            }

            // Prepare transaction data
            $transactionData = [
                'transaction_id' => 'TXN-' . strtoupper(uniqid()),
                'from_account_id' => $this->selectedAccount['id'],
                'amount' => $this->amount,
                'currency' => $this->selectedAccount['currency'],
                'fee_amount' => $this->transferFee + $this->swiftFee,
                'net_amount' => $this->totalAmount,
                'type' => 'transfer',
                'category' => 'banking',
                'subcategory' => 'international',
                'status' => 'pending',
                'description' => $this->description,
                'notes' => $this->notes,
                'transfer_method' => $this->transferMethod,
                'scheduled_at' => $this->transferType === 'scheduled' 
                    ? $this->scheduledDate . ' ' . $this->scheduledTime 
                    : null,
                'user_id' => auth()->id(),
            ];

            // Add beneficiary or manual details
            if ($this->useManualDetails) {
                // For manual transfers, store recipient details in metadata
                $transactionData['metadata'] = [
                    'swift_fee' => $this->swiftFee,
                    'transfer_fee' => $this->transferFee,
                    'manual_transfer' => true,
                    'recipient_details' => [
                        'name' => $this->recipientName,
                        'bank_name' => $this->recipientBankName,
                        'account_number' => $this->recipientAccountNumber,
                        'iban' => $this->recipientIban,
                        'swift_code' => $this->recipientSwiftCode,
                        'email' => $this->recipientEmail,
                        'phone' => $this->recipientPhone,
                        'address' => $this->recipientAddress,
                        'city' => $this->recipientCity,
                        'state' => $this->recipientState,
                        'postal_code' => $this->recipientPostalCode,
                        'country' => $this->recipientCountry,
                    ]
                ];
            } else {
                // For beneficiary transfers
                $transactionData['to_account_id'] = $this->selectedBeneficiary->account_id;
                $transactionData['beneficiary_id'] = $this->selectedBeneficiary->id;
                $transactionData['metadata'] = [
                    'swift_fee' => $this->swiftFee,
                    'transfer_fee' => $this->transferFee,
                    'manual_transfer' => false,
                ];
            }

            // Create transaction
            $transaction = Transaction::create($transactionData);

            // Update account balance - put amount in hold balance for pending transactions
            $account = Account::find($this->selectedAccount['id']);
            $account->increment('hold_balance', $this->totalAmount);
            $account->decrement('available_balance', $this->totalAmount);
            $account->decrement('balance', $this->totalAmount);

            // Check if user account status is transaction_declined after transaction creation
            $user = auth()->user();
            if ($user->customer_status === 'transaction_declined') {
                // Rollback the transaction
                $this->rollbackTransaction($transaction, $account);
                $this->isProcessing = false;
                return;
            }

            // Complete the transaction immediately (no code verification required)
            $transaction->complete();

            Flux::toast('Wire transfer completed successfully!', variant: 'success');
            
            // Redirect to success page
            return redirect()->route('user.transfer.success', ['transaction' => $transaction->id]);

        } catch (\Exception $e) {
            Flux::toast('Transfer failed: ' . $e->getMessage(), variant: 'danger');
        } finally {
            $this->isProcessing = false;
        }
    }

    protected function rollbackTransaction($transaction, $account)
    {
        try {
            // Update transaction status to cancelled
            $transaction->update(['status' => 'cancelled']);

            // Rollback account balance changes
            $account->decrement('hold_balance', $this->totalAmount);
            $account->increment('available_balance', $this->totalAmount);
            $account->increment('balance', $this->totalAmount);

            // Hide transfer summary and reset form
            $this->hideTransferSummary();
            $this->resetForm();

            // Show error message
            Flux::toast('Transaction declined. Please contact support.', variant: 'danger', duration: 10000);
        } catch (\Exception $e) {
            \Log::error('Transaction rollback failed: ' . $e->getMessage());
            Flux::toast('Transaction rollback failed. Please contact support.', variant: 'danger', duration: 10000);
        }
    }

    public function resetForm()
    {
        $this->fromAccountId = '';
        $this->beneficiaryId = '';
        $this->amount = '';
        $this->currency = 'USD';
        $this->transferType = 'instant';
        $this->transferMethod = 'online';
        $this->scheduledDate = '';
        $this->scheduledTime = '';
        $this->description = '';
        $this->notes = '';
        $this->selectedAccount = null;
        $this->selectedBeneficiary = null;
        $this->transferFee = 0;
        $this->swiftFee = 0;
        $this->totalAmount = 0;
        
        // Reset manual details
        $this->useManualDetails = false;
        $this->clearManualDetails();
    }

    public function getBeneficiariesProperty()
    {
        return Beneficiary::where('user_id', auth()->id())
            ->where('verification_status', 'verified')
            ->where('type', 'international')
            ->orderBy('name')
            ->get();
    }

    public function getUserAccountsProperty()
    {
        return $this->userAccounts;
    }

    public function render()
    {
        return view('livewire.user.banking.international-transfer');
    }
}
