<?php

namespace App\Livewire\User\Banking;

use App\Models\Transaction;
use App\Models\Account;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Layout;
use Flux;

#[Layout('components.layouts.app')]
class DepositHistory extends Component
{
    use WithPagination;

    // Filter properties
    public $statusFilter = '';
    public $accountFilter = '';
    public $dateFrom = '';
    public $dateTo = '';
    public $amountFrom = '';
    public $amountTo = '';

    // Statistics
    public $stats = [];
    public $userAccounts = [];

    protected $queryString = [
        'statusFilter' => ['except' => ''],
        'accountFilter' => ['except' => ''],
        'dateFrom' => ['except' => ''],
        'dateTo' => ['except' => ''],
        'amountFrom' => ['except' => ''],
        'amountTo' => ['except' => ''],
    ];

    public function mount()
    {
        $this->loadUserAccounts();
        $this->loadStats();
    }

    public function loadUserAccounts()
    {
        $user = Auth::user();
        $this->userAccounts = $user->accounts()
            ->where('status', 'active')
            ->get();
    }

    public function loadStats()
    {
        $user = Auth::user();
        $baseQuery = Transaction::where('user_id', $user->id)
            ->where('type', 'deposit');

        $this->stats = [
            'total_deposits' => $baseQuery->count(),
            'pending_deposits' => $baseQuery->where('status', 'pending')->count(),
            'completed_deposits' => $baseQuery->where('status', 'completed')->count(),
            'total_amount' => $baseQuery->where('status', 'completed')->sum('amount'),
            'pending_amount' => $baseQuery->where('status', 'pending')->sum('amount'),
        ];
    }

    public function getDepositsProperty()
    {
        $user = Auth::user();
        $query = Transaction::with(['toAccount'])
            ->where('user_id', $user->id)
            ->where('type', 'deposit');

        // Apply filters
        if ($this->statusFilter) {
            $query->where('status', $this->statusFilter);
        }

        if ($this->accountFilter) {
            $query->where('to_account_id', $this->accountFilter);
        }

        if ($this->dateFrom) {
            $query->whereDate('created_at', '>=', $this->dateFrom);
        }

        if ($this->dateTo) {
            $query->whereDate('created_at', '<=', $this->dateTo);
        }

        if ($this->amountFrom) {
            $query->where('amount', '>=', $this->amountFrom);
        }

        if ($this->amountTo) {
            $query->where('amount', '<=', $this->amountTo);
        }

        return $query->orderBy('created_at', 'desc')->paginate(15);
    }

    public function clearFilters()
    {
        $this->statusFilter = '';
        $this->accountFilter = '';
        $this->dateFrom = '';
        $this->dateTo = '';
        $this->amountFrom = '';
        $this->amountTo = '';
        $this->resetPage();
    }

    public function getStatusColor($status)
    {
        return match($status) {
            'pending' => 'bg-yellow-100 text-yellow-800',
            'processing' => 'bg-blue-100 text-blue-800',
            'completed' => 'bg-green-100 text-green-800',
            'failed' => 'bg-red-100 text-red-800',
            'cancelled' => 'bg-gray-100 text-gray-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    public function getStatusLabel($status)
    {
        return match($status) {
            'pending' => 'Pending Review',
            'processing' => 'Processing',
            'completed' => 'Completed',
            'failed' => 'Rejected',
            'cancelled' => 'Cancelled',
            default => 'Unknown',
        };
    }

    public function getStatusIcon($status)
    {
        return match($status) {
            'pending' => 'clock',
            'processing' => 'arrow-path',
            'completed' => 'check-circle',
            'failed' => 'x-circle',
            'cancelled' => 'minus-circle',
            default => 'question-mark-circle',
        };
    }

    public function getDepositTypeIcon($subcategory)
    {
        return match($subcategory) {
            'cheque_deposit' => 'document',
            'bank_transfer' => 'arrow-down-tray',
            'cash_deposit' => 'banknotes',
            'mobile_deposit' => 'device-phone-mobile',
            default => 'plus-circle',
        };
    }

    public function getDepositTypeLabel($subcategory)
    {
        return match($subcategory) {
            'cheque_deposit' => 'Cheque Deposit',
            'bank_transfer' => 'Bank Transfer',
            'cash_deposit' => 'Cash Deposit',
            'mobile_deposit' => 'Mobile Deposit',
            default => 'Deposit',
        };
    }

    public function render()
    {
        return view('livewire.user.banking.deposit-history', [
            'deposits' => $this->deposits,
        ]);
    }
}
