<?php

namespace App\Livewire\User\Banking;

use App\Models\Account;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use Flux;

#[Layout('components.layouts.app')]
class ChequeDeposit extends Component
{
    use WithFileUploads;

    // Form inputs
    #[Validate('required|exists:accounts,id')]
    public $accountId = '';

    #[Validate('required|numeric|min:0.01|max:100000')]
    public $amount = '';

    #[Validate('required|string|max:255')]
    public $bankName = '';

    #[Validate('required|string|max:50')]
    public $chequeNumber = '';

    #[Validate('nullable|string|max:20')]
    public $routingNumber = '';

    #[Validate('nullable|string|max:20')]
    public $accountNumber = '';

    #[Validate('nullable|string|max:500')]
    public $description = '';

    #[Validate('nullable|string|max:1000')]
    public $notes = '';

    // File uploads
    #[Validate('required|file|mimes:jpg,jpeg,png,pdf|max:10240')]
    public $frontImage;

    #[Validate('required|file|mimes:jpg,jpeg,png,pdf|max:10240')]
    public $backImage;

    // UI state
    public $showPreview = false;
    public $isProcessing = false;
    public $userAccounts = [];
    public $selectedAccount = null;

    protected $messages = [
        'accountId.required' => 'Please select an account to deposit into.',
        'accountId.exists' => 'Selected account is invalid.',
        'amount.required' => 'Please enter the cheque amount.',
        'amount.numeric' => 'Amount must be a valid number.',
        'amount.min' => 'Amount must be at least $0.01.',
        'amount.max' => 'Amount cannot exceed $100,000.',
        'bankName.required' => 'Please enter the bank name.',
        'chequeNumber.required' => 'Please enter the cheque number.',
        'frontImage.required' => 'Please upload the front image of the cheque.',
        'frontImage.mimes' => 'Front image must be a JPG, JPEG, PNG, or PDF file.',
        'frontImage.max' => 'Front image must not exceed 10MB.',
        'backImage.required' => 'Please upload the back image of the cheque.',
        'backImage.mimes' => 'Back image must be a JPG, JPEG, PNG, or PDF file.',
        'backImage.max' => 'Back image must not exceed 10MB.',
    ];

    public function mount()
    {
        $this->loadUserAccounts();
        $this->checkKycRequirement();
    }
    
    public function checkKycRequirement()
    {
        $isKycRequired = \App\Helpers\SettingsHelper::isKycRequired();
        
        if ($isKycRequired) {
            $user = auth()->user();
            $profile = $user->profile;
            
            if (!$profile || $profile->kyc_status !== 'verified') {
                if (!$user->hasRole(['super-admin', 'admin'])) {
                    $this->js("
                        setTimeout(() => {
                            if (window.Flux && window.Flux.toast) {
                                window.Flux.toast({
                                    variant: 'warning',
                                    heading: 'KYC Verification Required',
                                    text: 'Please complete your identity verification to access this feature.',
                                    duration: 6000
                                });
                            }
                            window.location.href = '" . route('user.kyc-verification') . "';
                        }, 2000);
                    ");
                }
            }
        }
    }

    public function loadUserAccounts()
    {
        $user = Auth::user();
        $this->userAccounts = $user->accounts()
            ->where('status', 'active')
            ->where('verification_status', 'verified')
            ->get();
    }

    public function updatedAccountId()
    {
        $this->selectedAccount = $this->userAccounts->firstWhere('id', $this->accountId);
    }

    public function updatedFrontImage()
    {
        $this->validate(['frontImage' => 'required|file|mimes:jpg,jpeg,png,pdf|max:10240']);
    }

    public function updatedBackImage()
    {
        $this->validate(['backImage' => 'required|file|mimes:jpg,jpeg,png,pdf|max:10240']);
    }

    public function showPreview()
    {
        $this->validate();
       
    }

    public function hidePreview()
    {
    
    }

    public function submitDeposit()
    {
        $this->validate();
        $this->isProcessing = true;

        try {
            $user = Auth::user();
            $account = Account::findOrFail($this->accountId);

            // Generate unique transaction ID
            $transactionId = 'TXN' . now()->format('Ymd') . Str::random(8);

            // Store cheque images
            $frontImagePath = $this->storeChequeImage($this->frontImage, 'front');
            $backImagePath = $this->storeChequeImage($this->backImage, 'back');

            // Create transaction record
            $transaction = Transaction::create([
                'transaction_id' => $transactionId,
                'reference_number' => 'CHQ-' . now()->format('YmdHis'),
                'user_id' => $user->id,
                'to_account_id' => $account->id,
                'type' => 'deposit',
                'category' => 'banking',
                'subcategory' => 'cheque_deposit',
                'amount' => $this->amount,
                'currency' => $account->currency,
                'fee_amount' => 0,
                'tax_amount' => 0,
                'net_amount' => $this->amount,
                'status' => 'pending',
                'verification_status' => 'pending',
                'description' => $this->description ?: "Cheque deposit from {$this->bankName}",
                'notes' => $this->notes,
                'payment_method' => 'cheque',
                'metadata' => [
                    'cheque_details' => [
                        'cheque_number' => $this->chequeNumber,
                        'bank_name' => $this->bankName,
                        'routing_number' => $this->routingNumber,
                        'account_number' => $this->accountNumber,
                        'front_image' => $frontImagePath,
                        'back_image' => $backImagePath,
                    ],
                    'upload_details' => [
                        'uploaded_at' => now()->toISOString(),
                        'ip_address' => request()->ip(),
                        'user_agent' => request()->userAgent(),
                    ],
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
                'risk_level' => $this->amount >= 10000 ? 'high' : 'low',
            ]);

            // Check for compliance flags
            $transaction->checkComplianceFlags();

            // Log the deposit submission
            activity()
                ->performedOn($transaction)
                ->log("Cheque deposit submitted: {$transactionId} for {$this->amount} {$account->currency}");

            // Send notification
            $notificationService = app(\App\Services\NotificationService::class);
            $notificationService->notifyChequeDeposit(
                auth()->user(),
                $this->amount,
                $account->currency,
                'submitted'
            );

            // Reset form
            $this->resetForm();

            Flux::toast(
                variant: 'success',
                heading: 'Cheque Deposit Submitted!',
                text: 'Your cheque deposit has been submitted for review. You will be notified once it\'s processed.'
            );

            // Redirect to deposit history
            return $this->redirect(route('user.deposit.history'), navigate: true);

        } catch (\Exception $e) {
            Flux::toast(
                variant: 'danger',
                heading: 'Deposit Failed',
                text: 'An error occurred while submitting your deposit. Please try again.'
            );

            \Log::error('Cheque Deposit Error: ' . $e->getMessage());
        } finally {
            $this->isProcessing = false;
        }
    }

  private function storeChequeImage($file, $type)
{
    $user = Auth::user();
    $extension = $file->getClientOriginalExtension();
    $filename = Str::uuid() . '-' . $type . '.' . $extension;

    // Define destination folder inside /public
    $destinationPath = public_path('cheque-deposits/' . $user->id);

    // Create folder if it doesn’t exist
    if (!file_exists($destinationPath)) {
        mkdir($destinationPath, 0755, true);
    }

    // If this is a Livewire TemporaryUploadedFile, use copy() instead of move()
    if (method_exists($file, 'getRealPath')) {
        copy($file->getRealPath(), $destinationPath . '/' . $filename);
    } else {
        $file->move($destinationPath, $filename);
    }

    // Return the relative public path to store in DB
    return 'cheque-deposits/' . $user->id . '/' . $filename;
}


    private function resetForm()
    {
        $this->accountId = '';
        $this->amount = '';
        $this->bankName = '';
        $this->chequeNumber = '';
        $this->routingNumber = '';
        $this->accountNumber = '';
        $this->description = '';
        $this->notes = '';
        $this->frontImage = null;
        $this->backImage = null;
        $this->selectedAccount = null;
        $this->resetValidation();
    }

    public function render()
    {
        return view('livewire.user.banking.cheque-deposit');
    }
}
