<?php

namespace App\Livewire\User;

use App\Models\Account;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use Flux;

#[Layout('components.layouts.app')]
class AccountManagement extends Component
{
    // Default limits (these should match your business requirements)
    private const DEFAULT_DAILY_TRANSFER_LIMIT = 10000;
    private const DEFAULT_MONTHLY_TRANSFER_LIMIT = 100000;
    private const DEFAULT_DAILY_WITHDRAWAL_LIMIT = 5000;
    private const DEFAULT_MONTHLY_WITHDRAWAL_LIMIT = 50000;
    private const MAX_ACCOUNTS = 3;

    public $user;
    public $accounts = [];
    public $selectedAccount = null;
    public $showAccountSettings = false;
    public $showAccountLimits = false;
    public $showAccountClosure = false;
    public $showAddAccount = false;
    public $isMasterAccount = false;
    
    // Account Settings
    #[Validate('required|string|max:255')]
    public string $account_name = '';
    
    #[Validate('nullable|string|max:500')]
    public string $description = '';
    
    #[Validate('boolean')]
    public bool $online_banking_enabled = true;
    
    #[Validate('boolean')]
    public bool $mobile_banking_enabled = true;
    
    #[Validate('boolean')]
    public bool $two_factor_required = true;
    
    // Account Limits
    #[Validate('required|numeric|min:0')]
    public string $daily_transfer_limit = '';
    
    #[Validate('required|numeric|min:0')]
    public string $monthly_transfer_limit = '';
    
    #[Validate('required|numeric|min:0')]
    public string $daily_withdrawal_limit = '';
    
    #[Validate('required|numeric|min:0')]
    public string $monthly_withdrawal_limit = '';

    public function mount()
    {
        $this->user = Auth::user();
        $this->loadAccounts();
    }

    public function loadAccounts()
    {
        $this->accounts = $this->user->accounts()->orderBy('created_at')->get();
        
        if ($this->accounts->isNotEmpty() && !$this->selectedAccount) {
            $this->selectedAccount = $this->accounts->first();
        }
    }

    public function selectAccount($accountId)
    {
        $this->selectedAccount = $this->accounts->find($accountId);
        $this->resetAccountForms();
        $this->showAccountSettings = false;
        $this->showAccountLimits = false;
        $this->showAccountClosure = false;
        $this->showAddAccount = false;
    }

    public function showSettings()
    {
        $this->showAccountSettings = true;
        $this->showAccountLimits = false;
        $this->showAccountClosure = false;
        $this->showAddAccount = false;
        $this->loadAccountSettings();
        $this->scrollToForm('account-settings-form');
    }

    public function showLimits()
    {
        $this->showAccountLimits = true;
        $this->showAccountSettings = false;
        $this->showAccountClosure = false;
        $this->showAddAccount = false;
        $this->loadAccountLimits();
        $this->scrollToForm('account-limits-form');
    }

    public function showClosure()
    {
        $this->showAccountClosure = true;
        $this->showAccountSettings = false;
        $this->showAccountLimits = false;
        $this->showAddAccount = false;
        $this->scrollToForm('account-closure-form');
    }

    public function openAddAccount()
    {
        $this->showAddAccount = true;
        $this->showAccountSettings = false;
        $this->showAccountLimits = false;
        $this->showAccountClosure = false;
        $this->scrollToForm('add-account-form');
    }

    private function scrollToForm($formId)
    {
        $this->js("
            setTimeout(() => {
                const form = document.getElementById('$formId');
                if (form) {
                    form.scrollIntoView({ 
                        behavior: 'smooth', 
                        block: 'start',
                        inline: 'nearest'
                    });
                }
            }, 100);
        ");
    }

    private function loadAccountSettings()
    {
        if ($this->selectedAccount) {
            $this->account_name = $this->selectedAccount->account_name;
            $this->description = $this->selectedAccount->description ?? '';
            $this->online_banking_enabled = $this->selectedAccount->online_banking_enabled;
            $this->mobile_banking_enabled = $this->selectedAccount->mobile_banking_enabled;
            $this->two_factor_required = $this->selectedAccount->two_factor_required;
            $this->isMasterAccount = $this->selectedAccount->is_master_account;
        }
    }

    private function loadAccountLimits()
    {
        if ($this->selectedAccount) {
            $this->daily_transfer_limit = $this->selectedAccount->daily_transfer_limit ?? '0';
            $this->monthly_transfer_limit = $this->selectedAccount->monthly_transfer_limit ?? '0';
            $this->daily_withdrawal_limit = $this->selectedAccount->daily_withdrawal_limit ?? '0';
            $this->monthly_withdrawal_limit = $this->selectedAccount->monthly_withdrawal_limit ?? '0';
        }
    }

    public function updateAccountSettings()
    {
        $this->validate([
            'account_name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
            'online_banking_enabled' => 'boolean',
            'mobile_banking_enabled' => 'boolean',
            'two_factor_required' => 'boolean',
        ]);

        if ($this->selectedAccount) {
            $this->selectedAccount->update([
                'account_name' => $this->account_name,
                'description' => $this->description,
                'online_banking_enabled' => $this->online_banking_enabled,
                'mobile_banking_enabled' => $this->mobile_banking_enabled,
                'two_factor_required' => $this->two_factor_required,
            ]);

            Flux::toast(
                variant: 'success',
                heading: 'Account Settings Updated!',
                text: 'Your account settings have been saved successfully.'
            );

            $this->showAccountSettings = false;
            $this->loadAccounts();
        }
    }

    public function updateAccountLimits()
    {
        $this->validate([
            'daily_transfer_limit' => 'required|numeric|min:0|max:' . self::DEFAULT_DAILY_TRANSFER_LIMIT,
            'monthly_transfer_limit' => 'required|numeric|min:0|max:' . self::DEFAULT_MONTHLY_TRANSFER_LIMIT,
            'daily_withdrawal_limit' => 'required|numeric|min:0|max:' . self::DEFAULT_DAILY_WITHDRAWAL_LIMIT,
            'monthly_withdrawal_limit' => 'required|numeric|min:0|max:' . self::DEFAULT_MONTHLY_WITHDRAWAL_LIMIT,
        ], [
            'daily_transfer_limit.max' => 'Daily transfer limit cannot exceed ' . number_format(self::DEFAULT_DAILY_TRANSFER_LIMIT) . ' ' . ($this->selectedAccount->currency ?? 'USD'),
            'monthly_transfer_limit.max' => 'Monthly transfer limit cannot exceed ' . number_format(self::DEFAULT_MONTHLY_TRANSFER_LIMIT) . ' ' . ($this->selectedAccount->currency ?? 'USD'),
            'daily_withdrawal_limit.max' => 'Daily withdrawal limit cannot exceed ' . number_format(self::DEFAULT_DAILY_WITHDRAWAL_LIMIT) . ' ' . ($this->selectedAccount->currency ?? 'USD'),
            'monthly_withdrawal_limit.max' => 'Monthly withdrawal limit cannot exceed ' . number_format(self::DEFAULT_MONTHLY_WITHDRAWAL_LIMIT) . ' ' . ($this->selectedAccount->currency ?? 'USD'),
        ]);

        if ($this->selectedAccount) {
            $this->selectedAccount->update([
                'daily_transfer_limit' => $this->daily_transfer_limit,
                'monthly_transfer_limit' => $this->monthly_transfer_limit,
                'daily_withdrawal_limit' => $this->daily_withdrawal_limit,
                'monthly_withdrawal_limit' => $this->monthly_withdrawal_limit,
            ]);

            Flux::toast(
                variant: 'success',
                heading: 'Account Limits Updated!',
                text: 'Your account limits have been updated successfully.'
            );

            $this->showAccountLimits = false;
            $this->loadAccounts();
        }
    }

    public function suspendAccount()
    {
        if ($this->selectedAccount) {
            
            if ($this->selectedAccount->is_master_account) {
                Flux::toast(
                    variant: 'danger',
                    heading: 'Cannot Suspend Primary Account!',
                    text: 'Your primary account cannot be suspended as it\'s essential for banking operations.'
                );
                return;
            }

            $this->selectedAccount->update(['status' => 'suspended']);

            Flux::toast(
                variant: 'warning',
                heading: 'Account Suspended!',
                text: 'Your account has been temporarily suspended.'
            );

            $this->showAccountClosure = false;
            $this->loadAccounts();
        }
    }

    public function activateAccount()
    {
        if ($this->selectedAccount) {
            $this->selectedAccount->update(['status' => 'active']);

            Flux::toast(
                variant: 'success',
                heading: 'Account Activated!',
                text: 'Your account has been activated successfully.'
            );

            $this->loadAccounts();
        }
    }

    public function closeAccount()
    {
        if ($this->selectedAccount) {
            // Check if account has zero balance
            if ($this->selectedAccount->balance > 0) {
                Flux::toast(
                    variant: 'error',
                    heading: 'Cannot Close Account!',
                    text: 'Account must have zero balance before closure.'
                );
                return;
            }

            $this->selectedAccount->update(['status' => 'closed']);

            Flux::toast(
                variant: 'success',
                heading: 'Account Closed!',
                text: 'Your account has been closed successfully.'
            );

            $this->showAccountClosure = false;
            $this->loadAccounts();
        }
    }

    public function redirectToAccountCreation()
    {
        return redirect()->route('account.create');
    }

    public function changeMasterAccount()
    {
        if ($this->selectedAccount && $this->selectedAccount->status === 'active') {
            $this->user->accounts()->update(['is_master_account' => false]);
            $this->selectedAccount->update(['is_master_account' => true]);
            
            Flux::toast(
                variant: 'success',
                heading: 'Primary Account Changed!',
                text: 'Your primary account has been updated successfully.'
            );
            
           sleep(0.5);
           $this->dispatch('$refresh');
           $this->loadAccounts();
        }
    }

    private function resetAccountForms()
    {
        $this->reset([
            'account_name', 'description', 'online_banking_enabled', 
            'mobile_banking_enabled', 'two_factor_required', 'isMasterAccount',
            'daily_transfer_limit', 'monthly_transfer_limit', 
            'daily_withdrawal_limit', 'monthly_withdrawal_limit'
        ]);
    }

    public function getAccountStatusColor($status)
    {
        return match($status) {
            'active' => 'bg-green-100 text-green-800',
            'pending' => 'bg-amber-100 text-amber-800',
            'suspended' => 'bg-red-100 text-red-800',
            'closed' => 'bg-gray-100 text-gray-800',
            default => 'bg-gray-100 text-gray-800'
        };
    }

    public function getAccountTypeColor($type)
    {
        return match($type) {
            'savings' => 'bg-blue-100 text-blue-800',
            'checking' => 'bg-purple-100 text-purple-800',
            'business' => 'bg-indigo-100 text-indigo-800',
            default => 'bg-gray-100 text-gray-800'
        };
    }

    public function getDefaultLimits()
    {
        return [
            'daily_transfer' => self::DEFAULT_DAILY_TRANSFER_LIMIT,
            'monthly_transfer' => self::DEFAULT_MONTHLY_TRANSFER_LIMIT,
            'daily_withdrawal' => self::DEFAULT_DAILY_WITHDRAWAL_LIMIT,
            'monthly_withdrawal' => self::DEFAULT_MONTHLY_WITHDRAWAL_LIMIT,
        ];
    }

    public function formatCurrency($amount, $currency = 'USD')
    {
        return $currency . number_format($amount, 2);
    }

    public function getAccountCountInfo()
    {
        $currentCount = $this->accounts->count();
        $maxAccounts = self::MAX_ACCOUNTS;
        $remainingAccounts = $maxAccounts - $currentCount;
        
        return [
            'current' => $currentCount,
            'max' => $maxAccounts,
            'remaining' => $remainingAccounts,
            'canAdd' => $remainingAccounts > 0
        ];
    }

    public function render()
    {
        return view('livewire.user.account-management');
    }
}
