<?php

namespace App\Livewire\Dashboard;

use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use App\Models\Account;
use App\Models\Transaction;
use App\Helpers\CurrencyHelper;
use App\Helpers\SettingsHelper;
use App\Services\ExchangeRateService;
use Flux\Flux;

class Overview extends Component
{
    public $totalBalance = 0;
    public $availableBalance = 0;
    public $pendingBalance = 0;
    public $recentTransactions = [];
    public $user;
    public $primaryAccount = null;
    public $currency = 'USD';
    public $virtualCard = null;
    public $balanceChangePercentage = 0;
    public $balanceChangeDirection = 'up';
    
    // Card Application Properties
    public $cardName = '';
    public $cardType = 'debit';
    public $cardBrand = 'visa';
    public $cardCurrency = 'USD';
    public $dailyLimit = null;
    public $monthlyLimit = null;
    
    // Card Funding Properties
    public $fundAmount = null;
    public $exchangeRate = null;

    protected $listeners = ['balanceUpdated'];

    public function mount()
    {
        $this->user = Auth::user();
        $this->loadDashboardData();
        
        if (session()->has('login_success')) {
            $this->js("
                setTimeout(() => {
                    if (window.Flux && window.Flux.toast) {
                        window.Flux.toast({
                            variant: 'success',
                            heading: 'Welcome back!',
                            text: 'You have successfully logged in.',
                            duration: 6000
                        });
                    }
                }, 100);
            ");
        }
    }

    public function loadDashboardData()
    {
        if ($this->user) {
            // Get user's accounts and calculate balances
            $accounts = $this->user->accounts;
            $primaryAccount = $accounts->where('is_master_account', true)->first();
            
            if ($primaryAccount) {
                $this->totalBalance = $primaryAccount->balance;
                $this->availableBalance = $primaryAccount->available_balance ?? $this->totalBalance;
                $this->pendingBalance = $primaryAccount->hold_balance ?? 0;
                $this->currency = $primaryAccount->currency;
                $this->primaryAccount = $primaryAccount;
                
                // Calculate balance change percentage
                $this->calculateBalanceChange();
            } else {
                $this->totalBalance = 0;
                $this->availableBalance = 0;
                $this->pendingBalance = 0;
                $this->currency = 'USD';
                $this->primaryAccount = null;
                $this->balanceChangePercentage = 0;
                $this->balanceChangeDirection = 'up';
            }
            
            $this->loadVirtualCard();
            $this->loadRecentTransactions();
        }
    }

    public function calculateBalanceChange()
    {
        if (!$this->primaryAccount) {
            $this->balanceChangePercentage = 0;
            $this->balanceChangeDirection = 'up';
            return;
        }

        // Get the balance from one month ago
        $oneMonthAgo = now()->subMonth();
        $previousBalance = $this->getBalanceFromDate($oneMonthAgo);
        
        if ($previousBalance > 0) {
            $change = $this->totalBalance - $previousBalance;
            $this->balanceChangePercentage = round(($change / $previousBalance) * 100, 1);
            $this->balanceChangeDirection = $change >= 0 ? 'up' : 'down';
        } else {
            $this->balanceChangePercentage = 0;
            $this->balanceChangeDirection = 'up';
        }
    }

    private function getBalanceFromDate($date)
    {
        // Get the account balance at the end of the specified date
        // This is a simplified calculation - in a real app, you'd track balance history
        $transactions = Transaction::where(function($query) {
                $query->where('from_account_id', $this->primaryAccount->id)
                      ->orWhere('to_account_id', $this->primaryAccount->id);
            })
            ->where('created_at', '<=', $date->endOfDay())
            ->get();

        $balance = 0;
        foreach ($transactions as $transaction) {
            // If this account is the recipient (to_account), add the amount
            if ($transaction->to_account_id == $this->primaryAccount->id) {
                $balance += $transaction->amount;
            }
            // If this account is the sender (from_account), subtract the amount
            elseif ($transaction->from_account_id == $this->primaryAccount->id) {
                $balance -= $transaction->amount;
            }
        }

        return $balance;
    }

    public function loadVirtualCard()
    {
        if ($this->user) {
            $this->virtualCard = $this->user->virtualCards()->first();
        }
    }

    public function loadRecentTransactions()
    {
        try {
            // Get the master account for the current user
            $masterAccount = $this->user->accounts()->where('is_master_account', true)->first();
            
            if (!$masterAccount) {
                $this->recentTransactions = collect([]);
                return;
            }

            $transactions = Transaction::where(function($query) use ($masterAccount) {
                // Show outgoing transactions from master account
                $query->where('from_account_id', $masterAccount->id)
                      // Show incoming transactions to master account, but exclude inter-account transfer receipts
                      ->orWhere(function($subQuery) use ($masterAccount) {
                          $subQuery->where('to_account_id', $masterAccount->id)
                                  ->where(function($subSubQuery) {
                                      // Include external incoming transactions
                                      $subSubQuery->whereNull('from_account_id')
                                                 ->orWhere('type', '!=', 'exchange');
                                  });
                      });
            })
            ->with(['fromAccount', 'toAccount', 'user'])
            ->latest()
            ->take(5)
            ->get();

            $this->recentTransactions = $transactions->map(function($transaction) use ($masterAccount) {
                return [
                    'id' => $transaction->id,
                    'transaction_id' => $transaction->transaction_id,
                    'reference' => $transaction->reference ?? $transaction->transaction_id ?? 'N/A',
                    'type' => $transaction->type,
                    'category' => $transaction->category,
                    'subcategory' => $transaction->subcategory,
                    'amount' => $transaction->amount,
                    'currency' => $transaction->currency,
                    'currency_symbol' => CurrencyHelper::getSymbol($transaction->currency),
                    'status' => $transaction->status,
                    'description' => $transaction->description,
                    'created_at' => $transaction->created_at,
                    'from_account_id' => $transaction->from_account_id,
                    'to_account_id' => $transaction->to_account_id,
                    'exchange_rate' => $transaction->exchange_rate,
                    'converted_amount' => $transaction->converted_amount,
                    'fee_amount' => $transaction->fee_amount,
                    'tax_amount' => $transaction->tax_amount,
                    'net_amount' => $transaction->net_amount,
                    'from_account' => optional($transaction->fromAccount)->account_number ?? 'N/A',
                    'to_account' => optional($transaction->toAccount)->account_number ?? 'N/A',
                    'sender' => optional($transaction->fromAccount)->user->name ?? optional($transaction->user)->name ?? 'N/A',
                    'beneficiary' => optional($transaction->toAccount)->user->name ?? optional($transaction->beneficiary)->name ?? 'N/A',
                ];
            });
        } catch (\Exception $e) {
            $this->recentTransactions = collect([]);
        }
    }

    public function getTransactionIcon($type, $fromAccountId, $toAccountId)
    {
        $masterAccount = $this->user->accounts()->where('is_master_account', true)->first();
        
        if ($type === 'exchange' || $type === 'transfer') {
            // For inter-account transfers and regular transfers, determine if it's incoming or outgoing
            if ($masterAccount) {
                if ($fromAccountId == $masterAccount->id) {
                    return 'arrow-up-right'; // Outgoing
                } elseif ($toAccountId == $masterAccount->id) {
                    return 'arrow-down-left'; // Incoming
                }
            }
        }
        
        // Default icons for other transaction types
        return match($type) {
            'debit', 'withdrawal', 'payment' => 'arrow-up-right',
            'credit', 'deposit', 'refund' => 'arrow-down-left',
            'transfer' => 'arrow-up-right', // Default to outgoing for transfers
            default => 'arrows-right-left'
        };
    }

    public function getTransactionIconColor($type, $fromAccountId, $toAccountId)
    {
        $masterAccount = $this->user->accounts()->where('is_master_account', true)->first();
        
        if ($type === 'exchange' || $type === 'transfer') {
            if ($masterAccount) {
                if ($fromAccountId == $masterAccount->id) {
                    return 'bg-red-100'; // Outgoing - red
                } elseif ($toAccountId == $masterAccount->id) {
                    return 'bg-green-100'; // Incoming - green
                }
            }
        }
        
        return match($type) {
            'debit', 'withdrawal', 'payment' => 'bg-red-100',
            'credit', 'deposit', 'refund' => 'bg-green-100',
            'transfer' => 'bg-red-100',
            default => 'bg-gray-100'
        };
    }

    public function getTransactionIconTextColor($type, $fromAccountId, $toAccountId)
    {
        $masterAccount = $this->user->accounts()->where('is_master_account', true)->first();
        
        if ($type === 'exchange' || $type === 'transfer') {
            if ($masterAccount) {
                if ($fromAccountId == $masterAccount->id) {
                    return 'text-red-600'; // Outgoing - red
                } elseif ($toAccountId == $masterAccount->id) {
                    return 'text-green-600'; // Incoming - green
                }
            }
        }
        
        return match($type) {
            'debit', 'withdrawal', 'payment' => 'text-red-600',
            'credit', 'deposit', 'refund' => 'text-green-600',
            'transfer' => 'text-red-600', // Default to red for outgoing transfers
            default => 'text-gray-600'
        };
    }

    public function getTransactionAmountColor($type, $fromAccountId, $toAccountId)
    {
        $masterAccount = $this->user->accounts()->where('is_master_account', true)->first();
        
        if ($type === 'exchange' || $type === 'transfer') {
            if ($masterAccount) {
                if ($fromAccountId == $masterAccount->id) {
                    return 'text-red-600'; // Outgoing - red
                } elseif ($toAccountId == $masterAccount->id) {
                    return 'text-green-600'; // Incoming - green
                }
            }
        }
        
        return match($type) {
            'debit', 'withdrawal', 'payment', 'card_payment' => 'text-red-600',
            'credit', 'deposit', 'refund' => 'text-green-600',
            'transfer' => 'text-red-600', // Default to red for outgoing transfers
            default => 'text-gray-600'
        };
    }

    public function getTransactionAmountPrefix($type, $fromAccountId, $toAccountId)
    {
        $masterAccount = $this->user->accounts()->where('is_master_account', true)->first();
        
        if ($type === 'exchange' || $type === 'transfer') {
            if ($masterAccount) {
                if ($fromAccountId == $masterAccount->id) {
                    return '-'; // Outgoing
                } elseif ($toAccountId == $masterAccount->id) {
                    return '+'; // Incoming
                }
            }
        }
        
        return match($type) {
            'debit', 'withdrawal', 'payment', 'card_payment' => '-',
            'credit', 'deposit', 'refund' => '+',
            'transfer', 'card_payment' => '-', // Default to negative for outgoing transfers
            default => '±'
        };
    }

    public function getTransactionTypeLabel($type)
    {
        return match($type) {
            'exchange' => 'Inter-Account Transfer',
            'debit' => 'Debit',
            'credit' => 'Credit',
            'withdrawal' => 'Withdrawal',
            'deposit' => 'Deposit',
            'payment' => 'Payment',
            'card_payment' => 'Card Payment',
            'refund' => 'Refund',
            'transfer' => 'Transfer',
            default => ucfirst($type)
        };
    }

    public function getTransactionStatusColor($status)
    {
        return match($status) {
            'completed' => 'bg-green-100 text-green-800',
            'successful' => 'bg-green-100 text-green-800',
            'pending' => 'bg-yellow-100 text-yellow-800',
            'processing' => 'bg-blue-100 text-blue-800',
            'failed' => 'bg-red-100 text-red-800',
            'cancelled' => 'bg-gray-100 text-gray-800',
            default => 'bg-gray-100 text-gray-800'
        };
    }

    public function getTransactionStatusLabel($status)
    {
        return match($status) {
            'completed' => 'Completed',
            'successful' => 'Successful',
            'pending' => 'Pending',
            'processing' => 'Processing',
            'failed' => 'Failed',
            'cancelled' => 'Cancelled',
            default => ucfirst($status)
        };
    }

    public function getTransactionDescription($transaction)
    {
        $masterAccount = $this->user->accounts()->where('is_master_account', true)->first();
        
        // Check if this is a card funding transaction
        if ($transaction['type'] === 'card_payment' && isset($transaction['category']) && $transaction['category'] === 'card' && isset($transaction['subcategory']) && $transaction['subcategory'] === 'funding') {
            return 'Card Funding';
        }
        
        if ($transaction['type'] === 'exchange') {
            if ($masterAccount) {
                if ($transaction['from_account_id'] == $masterAccount->id) {
                    return 'Inter-Account Transfer Out';
                } elseif ($transaction['to_account_id'] == $masterAccount->id) {
                    return 'Inter-Account Transfer In';
                }
            }
        } elseif ($transaction['type'] === 'transfer') {
            if ($masterAccount) {
                if ($transaction['from_account_id'] == $masterAccount->id) {
                    return 'Transfer Out';
                } elseif ($transaction['to_account_id'] == $masterAccount->id) {
                    return 'Transfer In';
                }
            }
        }
        
        return $transaction['sender'] ?: $transaction['beneficiary'] ?: 'Bank Transfer';
    }

    public function balanceUpdated()
    {
        $this->loadDashboardData();
    }



    public function deleteCard()
    {
        if ($this->virtualCard) {
            $cardBrand = $this->virtualCard->card_brand;
            $this->virtualCard->delete();
            $this->loadVirtualCard();
            
            Flux::toast(ucfirst($cardBrand) . ' card deleted successfully.', variant: 'success');
        }
    }

    public function processCardApplication()
    {
        $this->validate([
            'cardName' => 'required|string|max:255',
            'cardType' => 'required|in:debit,credit,prepaid',
            'cardBrand' => 'required|in:visa,mastercard,amex',
            'cardCurrency' => 'required|string|size:3',
        ]);

        try {
            // Generate card number (simplified for demo)
            $cardNumber = $this->generateCardNumber();
            $cvv = str_pad(rand(100, 999), 3, '0', STR_PAD_LEFT);
            $expiryDate = now()->addYears(3);

            $virtualCard = \App\Models\VirtualCard::create([
                'user_id' => $this->user->id,
                'account_id' => $this->primaryAccount->id,
                'card_number' => $cardNumber,
                'card_type' => $this->cardType,
                'card_brand' => $this->cardBrand,
                'card_name' => $this->cardName,
                'cvv' => $cvv,
                'expiry_date' => $expiryDate,
                'currency' => $this->cardCurrency,
                'current_balance' => 0,
                'daily_limit' => 0,
                'monthly_limit' => 0,
                'status' => 'active',
                'activated_at' => now(),
            ]);

            $this->loadVirtualCard();
            $this->resetCardApplicationForm();
            
            // Send notification
            $notificationService = app(\App\Services\NotificationService::class);
            $notificationService->notifyVirtualCardCreated(
                $this->user,
                ucfirst($this->cardBrand) . ' ' . ucfirst($this->cardType),
                substr($cardNumber, -4),
                [
                    'card_brand' => $this->cardBrand,
                    'card_type' => $this->cardType,
                    'card_name' => $this->cardName,
                    'currency' => $this->cardCurrency,
                    'expiry_date' => $expiryDate->format('m/Y'),
                    'daily_limit' => $this->dailyLimit,
                    'monthly_limit' => $this->monthlyLimit,
                    'account_number' => $this->primaryAccount->account_number,
                ]
            );
            
            Flux::toast(ucfirst($this->cardBrand) . ' card created successfully!', variant: 'success');
            
            // Refresh the page after successful card creation
            $this->js('setTimeout(() => { window.location.reload(); }, 1500);');

        } catch (\Exception $e) {
            Flux::toast('Failed to create ' . ucfirst($this->cardBrand) . ' card. Please try again.', variant: 'danger');
        }
    }

    public function updatedFundAmount()
    {
        if ($this->fundAmount && $this->virtualCard) {
            $this->loadExchangeRate();
        }
    }

    public function processCardFunding()
    {
        $this->validate([
            'fundAmount' => 'required|numeric|min:0.01|max:' . $this->availableBalance,
        ]);

        if (!$this->virtualCard) {
            Flux::toast('No card found.', variant: 'danger');
            return;
        }

        try {
            // Get exchange rate
            $this->loadExchangeRate();
            
            if (!$this->exchangeRate) {
                Flux::toast('Unable to get exchange rate. Please try again.', variant: 'danger');
                return;
            }

            $convertedAmount = $this->fundAmount * $this->exchangeRate;

            $transaction = \App\Models\Transaction::create([
                'transaction_id' => 'CARD-FUND-' . now()->format('YmdHis') . '-' . rand(1000, 9999),
                'reference_number' => 'CARD-FUND-' . now()->format('YmdHis'),
                'user_id' => $this->user->id,
                'from_account_id' => $this->primaryAccount->id,
                'to_account_id' => null, 
                'type' => 'card_payment',
                'category' => 'card',
                'subcategory' => 'funding',
                'amount' => $this->fundAmount,
                'currency' => $this->currency,
                'converted_amount' => $convertedAmount,
                'exchange_rate' => $this->exchangeRate,
                'net_amount' => $convertedAmount,
                'status' => 'completed',
                'description' => 'Fund ' . ucfirst($this->virtualCard->card_brand) . ' card',
                'completed_at' => now(),
            ]);

            // Update account balance
            $this->primaryAccount->decrement('balance', $this->fundAmount);
            $this->primaryAccount->decrement('available_balance', $this->fundAmount);

            // Update card balance
            $this->virtualCard->increment('current_balance', $convertedAmount);

            $this->loadDashboardData();
            $this->resetCardFundingForm();
            
            // Send notification
            $notificationService = app(\App\Services\NotificationService::class);
            $notificationService->notifyInterAccountTransfer(
                $this->user,
                $convertedAmount,
                $this->virtualCard->currency,
                'Master Account',
                ucfirst($this->virtualCard->card_brand) . ' Card'
            );
            
            Flux::toast(ucfirst($this->virtualCard->card_brand) . ' card funded successfully!', variant: 'success');
            
            // Refresh the page after successful card funding
            $this->js('setTimeout(() => { window.location.reload(); }, 1500);');

        } catch (\Exception $e) {
            Flux::toast('Failed to fund ' . ucfirst($this->virtualCard->card_brand) . ' card. Please try again.', variant: 'danger');
        }
    }

    private function generateCardNumber()
    {
        // Generate a realistic-looking card number based on brand
        $prefixes = [
            'visa' => ['4'],
            'mastercard' => ['51', '52', '53', '54', '55'],
            'amex' => ['34', '37']
        ];

        $prefix = $prefixes[$this->cardBrand][array_rand($prefixes[$this->cardBrand])];
        $length = $this->cardBrand === 'amex' ? 15 : 16;
        
        $number = $prefix;
        for ($i = strlen($prefix); $i < $length - 1; $i++) {
            $number .= rand(0, 9);
        }
        
        return $number;
    }

    private function resetCardApplicationForm()
    {
        $this->cardName = '';
        $this->cardType = 'debit';
        $this->cardBrand = 'visa';
        $this->cardCurrency = 'USD';
        $this->dailyLimit = null;
        $this->monthlyLimit = null;
    }

    private function resetCardFundingForm()
    {
        $this->fundAmount = null;
        $this->exchangeRate = null;
    }

    private function loadExchangeRate()
    {
        // Simplified exchange rate - in real app, you'd call an API
        $rates = [
            'USD_EUR' => 0.85,
            'USD_GBP' => 0.73,
            'EUR_USD' => 1.18,
            'EUR_GBP' => 0.86,
            'GBP_USD' => 1.37,
            'GBP_EUR' => 1.16,
        ];

        $key = $this->currency . '_' . $this->virtualCard->currency;
        $this->exchangeRate = $rates[$key] ?? 1.0;
    }

    public function render()
    {
        return view('livewire.dashboard.overview', [
            'currencySymbol' => CurrencyHelper::getSymbol($this->currency),
            'primaryAccount' => $this->primaryAccount,
            'swiftCode' => SettingsHelper::bank('swift_code'),
        ]);
    }
}
