<?php

namespace App\Livewire\Auth;

use App\Models\Account;
use App\Models\User;
use App\Helpers\SettingsHelper;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use Flux;

#[Layout('components.layouts.app')]
class AccountCreation extends Component
{
    private const MAX_ACCOUNTS = 3;

    #[Validate('required|in:savings,checking,business,offshore')]
    public string $account_type = 'savings';
    
    #[Validate('required|string|max:255')]
    public string $account_name = '';
    
    #[Validate('required|in:USD,EUR,GBP')]
    public string $currency = 'USD';
    
    #[Validate('nullable|string|max:500')]
    public string $description = '';

    public bool $accountCreated = false;
    public ?Account $createdAccount = null;

    public function mount()
    {
        $user = Auth::user();
        
        // Check if user has reached account limit
        $currentAccountCount = $user->accounts()->count();
        if ($currentAccountCount >= self::MAX_ACCOUNTS) {
            Flux::toast(
                variant: 'danger',
                heading: 'Account Limit Reached!',
                text: 'You have reached the maximum limit of ' . self::MAX_ACCOUNTS . ' accounts.'
            );
            return $this->redirect(route('user.account.settings'), navigate: true);
        }
        
        $this->account_name = $user->name;
    }

    public function updatedAccountType()
    {
        $user = Auth::user();
        
        $this->account_name = $user->name;
    }

    public function createAccount()
    {
        $this->validate();

        $user = Auth::user();

        // Check if user is active
        if ($user->customer_status !== 'active') {
            Flux::toast(
                variant: 'danger',
                heading: 'Account Creation Not Allowed!',
                text: 'Your account must be active before creating banking accounts. Please complete your profile verification first.'
            );
            return;
        }

        // Check if email verification is required and completed
        if (SettingsHelper::isEmailVerificationEnabled() && !$user->hasVerifiedEmail()) {
            Flux::toast(
                variant: 'danger',
                heading: 'Email Verification Required!',
                text: 'Please verify your email address before creating banking accounts.'
            );
            return;
        }

        // Check account limit
        $currentAccountCount = $user->accounts()->count();
        if ($currentAccountCount >= self::MAX_ACCOUNTS) {
            Flux::toast(
                variant: 'danger',
                heading: 'Account Limit Reached!',
                text: 'You have reached the maximum limit of ' . self::MAX_ACCOUNTS . ' accounts.'
            );
            return;
        }

        // Generate unique account number
        $accountNumber = $this->generateAccountNumber();

        // Get limits from settings
        $limits = $this->getAccountLimits($this->account_type);

        // Check if this will be the master account (first account)
        $isMasterAccount = !$user->accounts()->exists();
        
        // Create the account
        $account = Account::create([
            'user_id' => $user->id,
            'account_number' => $accountNumber,
            'account_type' => $this->account_type,
            'account_name' => $this->account_name,
            'currency' => $this->currency,
            'description' => $this->description,
            'balance' => 0.00,
            'available_balance' => 0.00,
            'pending_balance' => 0.00,
            'overdraft_limit' => 0.00,
            'daily_transfer_limit' => $limits['daily_transfer'],
            'daily_withdrawal_limit' => $limits['daily_withdrawal'],
            'monthly_transfer_limit' => $limits['monthly_transfer'],
            'monthly_withdrawal_limit' => $limits['monthly_withdrawal'],
            'status' => 'pending',
            'verification_status' => 'pending',
            'online_banking_enabled' => true,
            'mobile_banking_enabled' => true,
            'two_factor_required' => true,
            'risk_level' => 'medium',
            'aml_monitoring_enabled' => true,
            'is_master_account' => $isMasterAccount,
        ]);

        $this->createdAccount = $account;
        $this->accountCreated = true;

        // Send notification for account creation
        $notificationService = app(\App\Services\NotificationService::class);
        $notificationService->notifyAccountCreated(
            user: $user,
            accountNumber: $accountNumber,
            accountType: $this->account_type,
            currency: $this->currency,
            isMasterAccount: $isMasterAccount
        );

        Flux::toast(
            variant: 'success',
            heading: 'Account Created Successfully!',
            text: 'Your banking account has been created and is pending verification.'
        );
    }

    public function proceedToDashboard()
    {
        return $this->redirect(route('dashboard'), navigate: true);
    }

    private function generateAccountNumber(): string
    {
        do {
            // Generate 10-digit account number starting with bank code
            $accountNumber = '5001' . str_pad(random_int(100000, 999999), 6, '0', STR_PAD_LEFT);
        } while (Account::where('account_number', $accountNumber)->exists());

        return $accountNumber;
    }

    private function getAccountLimits(string $accountType): array
    {
        // Get base limits from settings
        $baseLimits = SettingsHelper::getTransactionLimits();
        
        // Apply multipliers based on account type
        $multipliers = match($accountType) {
            'savings' => [
                'daily_transfer' => 0.5,    // 50% of base
                'monthly_transfer' => 0.5,  // 50% of base
                'daily_withdrawal' => 0.2,  // 20% of base
                'monthly_withdrawal' => 0.2, // 20% of base
            ],
            'checking' => [
                'daily_transfer' => 1.0,    // 100% of base
                'monthly_transfer' => 1.0,  // 100% of base
                'daily_withdrawal' => 0.4,  // 40% of base
                'monthly_withdrawal' => 0.4, // 40% of base
            ],
            'business' => [
                'daily_transfer' => 2.5,    // 250% of base
                'monthly_transfer' => 5.0,  // 500% of base
                'daily_withdrawal' => 1.0,  // 100% of base
                'monthly_withdrawal' => 1.0, // 100% of base
            ],
            'offshore' => [
                'daily_transfer' => 5.0,    // 500% of base
                'monthly_transfer' => 10.0, // 1000% of base
                'daily_withdrawal' => 2.0,  // 200% of base
                'monthly_withdrawal' => 2.0, // 200% of base
            ],
            default => [
                'daily_transfer' => 0.5,
                'monthly_transfer' => 0.5,
                'daily_withdrawal' => 0.2,
                'monthly_withdrawal' => 0.2,
            ]
        };

        return [
            'daily_transfer' => $baseLimits['daily_transfer_limit'] * $multipliers['daily_transfer'],
            'monthly_transfer' => $baseLimits['monthly_transfer_limit'] * $multipliers['monthly_transfer'],
            'daily_withdrawal' => $baseLimits['daily_withdrawal_limit'] * $multipliers['daily_withdrawal'],
            'monthly_withdrawal' => $baseLimits['monthly_withdrawal_limit'] * $multipliers['monthly_withdrawal'],
        ];
    }

    public function getCurrencySymbolProperty(): string
    {
        return match($this->currency) {
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            default => '$'
        };
    }

    public function getCurrencyNameProperty(): string
    {
        return match($this->currency) {
            'USD' => 'US Dollar',
            'EUR' => 'Euro',
            'GBP' => 'British Pound',
            default => 'US Dollar'
        };
    }

    public function render()
    {
        return view('livewire.auth.account-creation');
    }
}
