<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use App\Helpers\SettingsHelper;
use Flux;

class KycVerificationMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Skip KYC check for guests
        if (!auth()->check()) {
            return $next($request);
        }

        $user = auth()->user();

        // Skip KYC check if KYC is not required
        if (!SettingsHelper::isKycRequired()) {
            return $next($request);
        }

        // Skip KYC check for admin users
        if ($user->hasRole(['super-admin', 'admin', 'manager'])) {
            return $next($request);
        }

        // Skip KYC check for KYC verification page itself to avoid redirect loops
        if ($request->routeIs('user.kyc-verification')) {
            return $next($request);
        }

        // Skip KYC check for profile completion page
        if ($request->routeIs('profile.complete')) {
            return $next($request);
        }

        // Skip KYC check for logout and other essential routes
        if ($request->routeIs(['logout', 'verification.notice', 'verification.verify', 'password.request', 'password.reset'])) {
            return $next($request);
        }

        // Check if user has completed KYC verification
        $userProfile = $user->profile;
        if (!$userProfile || $userProfile->kyc_status !== 'verified') {
            // Redirect to KYC verification page with session data
            return redirect()->route('user.kyc-verification')->with([
                'kyc_required' => true,
                'kyc_variant' => 'warning',
                'kyc_heading' => 'KYC Verification Required',
                'kyc_text' => 'Please complete your identity verification to access this feature.'
            ]);
        }

        return $next($request);
    }
}
