<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\VirtualCard;
use App\Models\User;
use App\Models\Account;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class VirtualCardController extends Controller
{
    /**
     * Display a listing of virtual cards
     */
    public function index(Request $request)
    {
        $query = VirtualCard::with(['user', 'account']);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('card_number', 'like', "%{$search}%")
                  ->orWhere('card_name', 'like', "%{$search}%")
                  ->orWhereHas('user', function ($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by card type
        if ($request->filled('card_type')) {
            $query->where('card_type', $request->card_type);
        }

        // Filter by card brand
        if ($request->filled('card_brand')) {
            $query->where('card_brand', $request->card_brand);
        }

        // Filter by currency
        if ($request->filled('currency')) {
            $query->where('currency', $request->currency);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Sort
        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        $query->orderBy($sortBy, $sortDirection);

        $cards = $query->paginate(10);

        return view('admin.virtual-cards.index', compact('cards'));
    }

    /**
     * Show the form for creating a new virtual card
     */
    public function create()
    {
        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'currency', 'user_id')
            ->with('user:id,name')
            ->get();

        return view('admin.virtual-cards.create', compact('users', 'accounts'));
    }

    /**
     * Store a newly created virtual card
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'account_id' => 'required|exists:accounts,id',
            'card_type' => 'required|in:debit,credit,prepaid',
            'card_brand' => 'required|in:visa,mastercard,amex',
            'card_name' => 'nullable|string|max:255',
            'expiry_date' => 'required|date|after:today',
            'currency' => 'required|string|size:3',
            'credit_limit' => 'nullable|numeric|min:0',
            'daily_limit' => 'nullable|numeric|min:0',
            'monthly_limit' => 'nullable|numeric|min:0',
            'online_payments' => 'boolean',
            'atm_withdrawals' => 'boolean',
            'international_usage' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            // Generate virtual card number
            $cardNumber = $this->generateCardNumber($request->card_brand);
            
            // Generate CVV
            $cvv = str_pad(rand(100, 999), 3, '0', STR_PAD_LEFT);

            $card = VirtualCard::create([
                'user_id' => $request->user_id,
                'account_id' => $request->account_id,
                'card_number' => $cardNumber,
                'card_type' => $request->card_type,
                'card_brand' => $request->card_brand,
                'card_name' => $request->card_name,
                'cvv' => $cvv,
                'expiry_date' => $request->expiry_date,
                'currency' => $request->currency,
                'credit_limit' => $request->credit_limit,
                'current_balance' => 0,
                'daily_limit' => $request->daily_limit,
                'monthly_limit' => $request->monthly_limit,
                'status' => 'active',
                'activated_at' => now(),
                'online_payments' => $request->boolean('online_payments'),
                'atm_withdrawals' => $request->boolean('atm_withdrawals'),
                'international_usage' => $request->boolean('international_usage'),
            ]);

            return redirect()->route('admin.virtual-cards.show', $card)
                ->with('success', 'Virtual card created successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create virtual card: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified virtual card
     */
    public function show(VirtualCard $virtualCard)
    {
        $virtualCard->load(['user', 'account']);
        
        // Get recent transactions for this virtual card
        $recentTransactions = $virtualCard->transactions()
            ->latest()
            ->limit(10)
            ->get();

        return view('admin.virtual-cards.show', compact('virtualCard', 'recentTransactions'));
    }

    /**
     * Show the form for editing the virtual card
     */
    public function edit(VirtualCard $virtualCard)
    {
        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'currency', 'user_id')
            ->with('user:id,name')
            ->get();

        return view('admin.virtual-cards.edit', compact('virtualCard', 'users', 'accounts'));
    }

    /**
     * Update the specified virtual card
     */
    public function update(Request $request, VirtualCard $virtualCard)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'account_id' => 'required|exists:accounts,id',
            'card_number' => 'required|string|min:15|max:16|unique:virtual_cards,card_number,' . $virtualCard->id,
            'cvv' => 'required|string|min:3|max:4',
            'card_type' => 'required|in:debit,credit,prepaid',
            'card_brand' => 'required|in:visa,mastercard,amex',
            'card_name' => 'nullable|string|max:255',
            'expiry_date' => 'required|date|after:today',
            'currency' => 'required|string|size:3',
            'credit_limit' => 'nullable|numeric|min:0',
            'current_balance' => 'nullable|numeric|min:0',
            'daily_limit' => 'nullable|numeric|min:0',
            'monthly_limit' => 'nullable|numeric|min:0',
            'online_payments' => 'boolean',
            'atm_withdrawals' => 'boolean',
            'international_usage' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $virtualCard->update([
                'user_id' => $request->user_id,
                'account_id' => $request->account_id,
                'card_number' => $request->card_number,
                'cvv' => $request->cvv,
                'card_type' => $request->card_type,
                'card_brand' => $request->card_brand,
                'card_name' => $request->card_name,
                'expiry_date' => $request->expiry_date,
                'currency' => $request->currency,
                'credit_limit' => $request->credit_limit,
                'current_balance' => $request->current_balance,
                'daily_limit' => $request->daily_limit,
                'monthly_limit' => $request->monthly_limit,
                'online_payments' => $request->boolean('online_payments'),
                'atm_withdrawals' => $request->boolean('atm_withdrawals'),
                'international_usage' => $request->boolean('international_usage'),
            ]);

            return redirect()->route('admin.virtual-cards.show', $virtualCard)
                ->with('success', 'Virtual card updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update virtual card: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified virtual card
     */
    public function destroy(VirtualCard $virtualCard)
    {
        try {
            $virtualCard->delete();
            return redirect()->route('admin.virtual-cards.index')
                ->with('success', 'Virtual card deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete virtual card: ' . $e->getMessage());
        }
    }

    /**
     * Block a virtual card
     */
    public function block(VirtualCard $virtualCard)
    {
        try {
            $virtualCard->update(['status' => 'blocked']);
            return redirect()->back()
                ->with('success', 'Virtual card blocked successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to block virtual card: ' . $e->getMessage());
        }
    }

    /**
     * Unblock a virtual card
     */
    public function unblock(VirtualCard $virtualCard)
    {
        try {
            $virtualCard->update(['status' => 'active']);
            return redirect()->back()
                ->with('success', 'Virtual card unblocked successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to unblock virtual card: ' . $e->getMessage());
        }
    }

    /**
     * Deactivate a virtual card
     */
    public function deactivate(VirtualCard $virtualCard)
    {
        try {
            $virtualCard->update(['status' => 'inactive']);
            return redirect()->back()
                ->with('success', 'Virtual card deactivated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to deactivate virtual card: ' . $e->getMessage());
        }
    }

    /**
     * Activate a virtual card
     */
    public function activate(VirtualCard $virtualCard)
    {
        try {
            $virtualCard->update([
                'status' => 'active',
                'activated_at' => now()
            ]);
            return redirect()->back()
                ->with('success', 'Virtual card activated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to activate virtual card: ' . $e->getMessage());
        }
    }

    /**
     * Generate new CVV for a virtual card
     */
    public function regenerateCvv(VirtualCard $virtualCard)
    {
        try {
            $newCvv = str_pad(rand(100, 999), 3, '0', STR_PAD_LEFT);
            $virtualCard->update(['cvv' => $newCvv]);
            
            return redirect()->back()
                ->with('success', 'CVV regenerated successfully. New CVV: ' . $newCvv);
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to regenerate CVV: ' . $e->getMessage());
        }
    }

    /**
     * Get virtual card statistics
     */
    public function statistics()
    {
        $stats = [
            'total_cards' => VirtualCard::count(),
            'active_cards' => VirtualCard::where('status', 'active')->count(),
            'blocked_cards' => VirtualCard::where('status', 'blocked')->count(),
            'inactive_cards' => VirtualCard::where('status', 'inactive')->count(),
            'expired_cards' => VirtualCard::where('expiry_date', '<', now())->count(),
            'total_credit_limit' => VirtualCard::sum('credit_limit'),
            'total_current_balance' => VirtualCard::sum('current_balance'),
            'total_spent' => VirtualCard::sum('total_spent'),
            'cards_by_type' => VirtualCard::selectRaw('card_type, COUNT(*) as count')
                ->groupBy('card_type')
                ->pluck('count', 'card_type'),
            'cards_by_brand' => VirtualCard::selectRaw('card_brand, COUNT(*) as count')
                ->groupBy('card_brand')
                ->pluck('count', 'card_brand'),
            'cards_by_currency' => VirtualCard::selectRaw('currency, COUNT(*) as count')
                ->groupBy('currency')
                ->pluck('count', 'currency'),
        ];

        return view('admin.virtual-cards.statistics', compact('stats'));
    }

    /**
     * Generate a virtual card number based on brand
     */
    private function generateCardNumber($brand)
    {
        $prefixes = [
            'visa' => ['4'],
            'mastercard' => ['51', '52', '53', '54', '55'],
            'amex' => ['34', '37']
        ];

        $prefix = $prefixes[$brand][array_rand($prefixes[$brand])];
        $length = $brand === 'amex' ? 15 : 16;
        
        $number = $prefix;
        for ($i = strlen($prefix); $i < $length - 1; $i++) {
            $number .= rand(0, 9);
        }

        // Add check digit using Luhn algorithm
        $number .= $this->calculateLuhnCheckDigit($number);

        // Ensure uniqueness
        while (VirtualCard::where('card_number', $number)->exists()) {
            $number = substr($number, 0, -1) . rand(0, 9);
        }

        return $number;
    }

    /**
     * Calculate Luhn check digit
     */
    private function calculateLuhnCheckDigit($number)
    {
        $sum = 0;
        $alternate = false;
        
        for ($i = strlen($number) - 1; $i >= 0; $i--) {
            $digit = (int) $number[$i];
            
            if ($alternate) {
                $digit *= 2;
                if ($digit > 9) {
                    $digit = ($digit % 10) + 1;
                }
            }
            
            $sum += $digit;
            $alternate = !$alternate;
        }
        
        return (10 - ($sum % 10)) % 10;
    }
}