<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\TransactionGeneratorService;
use App\Models\Account;
use App\Models\User;
use Illuminate\Http\Request;
use Carbon\Carbon;

class TransactionGeneratorController extends Controller
{
    protected $transactionGenerator;

    public function __construct(TransactionGeneratorService $transactionGenerator)
    {
        $this->transactionGenerator = $transactionGenerator;
    }

    /**
     * Display the transaction generator form
     */
    public function index()
    {
        $accounts = Account::with('user')
            ->where('status', 'active')
            ->orderBy('account_name')
            ->get();

        $users = User::select('id', 'name', 'email')
            ->orderBy('name')
            ->get();

        return view('admin.transaction-generator.index', compact('accounts', 'users'));
    }

    /**
     * Generate transactions
     */
    public function generate(Request $request)
    {
        $request->validate([
            'account_ids' => 'required|array|min:1',
            'account_ids.*' => 'exists:accounts,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'transaction_count' => 'required|integer|min:1|max:1000',
            'direction' => 'required|in:incoming,outgoing,mixed',
            'generate_all' => 'boolean'
        ]);

        try {
            $startDate = Carbon::parse($request->start_date);
            $endDate = Carbon::parse($request->end_date);
            $count = $request->transaction_count;
            $direction = $request->direction;
            $accountIds = $request->account_ids;

            // Generate transactions
            if ($request->generate_all) {
                $allAccounts = Account::where('status', 'active')->pluck('id')->toArray();
                $transactions = $this->transactionGenerator->generateBulkTransactions(
                    $allAccounts,
                    $startDate,
                    $endDate,
                    $count,
                    $direction
                );
                $message = "Generated " . count($transactions) . " transactions across all active accounts";
            } else {
                $transactions = $this->transactionGenerator->generateBulkTransactions(
                    $accountIds,
                    $startDate,
                    $endDate,
                    $count,
                    $direction
                );
                $message = "Generated " . count($transactions) . " transactions for selected accounts";
            }

            // Log the generation
            activity()
                ->log("Transaction generator used: {$message}");

            return redirect()->route('admin.transaction-generator.index')
                ->with('success', $message);

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to generate transactions: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Get account details for AJAX
     */
    public function getAccountDetails(Account $account)
    {
        return response()->json([
            'id' => $account->id,
            'account_number' => $account->account_number,
            'account_name' => $account->account_name,
            'currency' => $account->currency,
            'balance' => $account->balance,
            'user_name' => $account->user->name,
            'user_email' => $account->user->email,
        ]);
    }

    /**
     * Preview transaction generation (without creating)
     */
    public function preview(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:accounts,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'transaction_count' => 'required|integer|min:1|max:50',
            'direction' => 'required|in:incoming,outgoing,mixed'
        ]);

        try {
            $account = Account::findOrFail($request->account_id);
            $startDate = Carbon::parse($request->start_date);
            $endDate = Carbon::parse($request->end_date);
            $count = min($request->transaction_count, 10); // Limit preview to 10 transactions
            $direction = $request->direction;

            // Generate preview data (without saving)
            $previewData = $this->generatePreviewData($account, $startDate, $endDate, $count, $direction);

            return response()->json([
                'success' => true,
                'preview' => $previewData
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Generate preview data without saving
     */
    private function generatePreviewData(Account $account, Carbon $startDate, Carbon $endDate, int $count, string $direction): array
    {
        $transactions = [];
        $currentDate = $startDate->copy();
        
        // Transaction types to exclude (as requested)
        $transactionTypes = [
            'deposit' => ['wire_deposit', 'ach_deposit', 'bank_transfer_deposit', 'digital_wallet_deposit', 'interest_deposit', 'refund_deposit'],
            'withdrawal' => ['wire_withdrawal', 'ach_withdrawal', 'bank_transfer_withdrawal', 'digital_wallet_withdrawal'],
            'transfer' => ['wire_transfer', 'ach_transfer', 'swift_transfer', 'sepa_transfer', 'domestic_transfer'],
            'payment' => ['bill_payment', 'merchant_payment', 'utility_payment', 'loan_payment'],
            'refund' => ['merchant_refund', 'service_refund', 'fee_refund'],
            'fee' => ['maintenance_fee', 'transfer_fee', 'overdraft_fee', 'late_fee'],
            'interest' => ['savings_interest', 'deposit_interest', 'loan_interest']
        ];
        
        // Generate random dates within the range
        $dateRange = $endDate->diffInDays($startDate);
        $randomDates = [];
        
        for ($i = 0; $i < $count; $i++) {
            $randomDays = rand(0, $dateRange);
            $randomHours = rand(0, 23);
            $randomMinutes = rand(0, 59);
            
            $randomDates[] = $startDate->copy()->addDays($randomDays)
                ->setHour($randomHours)
                ->setMinute($randomMinutes)
                ->setSecond(0);
        }
        
        sort($randomDates);
        
        foreach ($randomDates as $transactionDate) {
            $type = array_rand($transactionTypes);
            $subcategory = $transactionTypes[$type][array_rand($transactionTypes[$type])];
            
            // Determine if this should be incoming or outgoing
            $isIncoming = $this->shouldBeIncoming($type, $direction);
            
            // Generate amount
            $amount = $this->generateAmount($type, $isIncoming);
            $feeAmount = $this->generateFee($type, $amount);
            $taxAmount = $this->generateTax($amount);
            $netAmount = $amount - $feeAmount - $taxAmount;
            
            $transactions[] = [
                'type' => $type,
                'subcategory' => $subcategory,
                'amount' => $amount,
                'fee_amount' => $feeAmount,
                'tax_amount' => $taxAmount,
                'net_amount' => $netAmount,
                'currency' => $account->currency,
                'direction' => $isIncoming ? 'incoming' : 'outgoing',
                'date' => $transactionDate->format('Y-m-d H:i:s'),
                'description' => ucfirst(str_replace('_', ' ', $subcategory))
            ];
        }
        
        return $transactions;
    }

    private function shouldBeIncoming(string $type, string $direction): bool
    {
        if ($direction === 'incoming') return true;
        if ($direction === 'outgoing') return false;
        
        $incomingTypes = ['deposit', 'refund', 'interest'];
        return in_array($type, $incomingTypes);
    }

    private function generateAmount(string $type, bool $isIncoming): float
    {
        $ranges = [
            'deposit' => [1000, 500000],
            'withdrawal' => [500, 100000],
            'transfer' => [250, 250000],
            'payment' => [100, 50000],
            'refund' => [50, 20000],
            'fee' => [25, 1000],
            'interest' => [10, 5000],
        ];
        
        $range = $ranges[$type] ?? [100, 10000];
        $amount = rand($range[0] * 100, $range[1] * 100) / 100;
        return round($amount, 2);
    }

    private function generateFee(string $type, float $amount): float
    {
        $feeRates = [
            'transfer' => 0.02,
            'withdrawal' => 0.015,
            'payment' => 0.01,
            'deposit' => 0.005,
        ];
        
        $rate = $feeRates[$type] ?? 0.01;
        $fee = max($amount * $rate, 2.50);
        return round($fee, 2);
    }

    private function generateTax(float $amount): float
    {
        return round($amount * 0.05, 2);
    }
}