<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class SystemLogsController extends Controller
{
    /**
     * Display a listing of system logs
     */
    public function index(Request $request)
    {
        $logFiles = $this->getLogFiles();
        $selectedFile = $request->get('file', $logFiles->first()['name'] ?? null);
        $logs = [];
        $pagination = null;

        if ($selectedFile) {
            $logs = $this->getLogEntries($selectedFile, $request);
            $pagination = $this->getPagination($selectedFile, $request);
        }

        return view('admin.logs.index', compact('logFiles', 'selectedFile', 'logs', 'pagination'));
    }

    /**
     * Display a specific log file
     */
    public function show(Request $request, $filename)
    {
        $logFiles = $this->getLogFiles();
        $logs = $this->getLogEntries($filename, $request);
        $pagination = $this->getPagination($filename, $request);

        return view('admin.logs.show', compact('logFiles', 'filename', 'logs', 'pagination'));
    }

    /**
     * Download a log file
     */
    public function download($filename)
    {
        $logPath = storage_path('logs/' . $filename);
        
        if (!File::exists($logPath)) {
            return redirect()->back()->with('error', 'Log file not found.');
        }

        return response()->download($logPath);
    }

    /**
     * Clear a log file
     */
    public function clear($filename)
    {
        $logPath = storage_path('logs/' . $filename);
        
        if (!File::exists($logPath)) {
            return redirect()->back()->with('error', 'Log file not found.');
        }

        try {
            File::put($logPath, '');
            return redirect()->back()->with('success', 'Log file cleared successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to clear log file: ' . $e->getMessage());
        }
    }

    /**
     * Get all log files
     */
    private function getLogFiles()
    {
        $logPath = storage_path('logs');
        $files = collect(File::files($logPath))
            ->map(function ($file) {
                return [
                    'name' => $file->getFilename(),
                    'size' => $this->formatBytes($file->getSize()),
                    'modified' => $file->getMTime(),
                    'path' => $file->getPathname()
                ];
            })
            ->sortByDesc('modified')
            ->values();

        return $files;
    }

    /**
     * Get log entries with pagination
     */
    private function getLogEntries($filename, Request $request)
    {
        $logPath = storage_path('logs/' . $filename);
        
        if (!File::exists($logPath)) {
            return collect();
        }

        $page = $request->get('page', 1);
        $perPage = 50;
        $offset = ($page - 1) * $perPage;

        $content = File::get($logPath);
        $lines = explode("\n", $content);
        
        // Filter by level if specified
        if ($request->filled('level')) {
            $lines = array_filter($lines, function ($line) use ($request) {
                return strpos($line, '[' . strtoupper($request->level) . ']') !== false;
            });
        }

        // Filter by search term
        if ($request->filled('search')) {
            $search = $request->search;
            $lines = array_filter($lines, function ($line) use ($search) {
                return stripos($line, $search) !== false;
            });
        }

        $totalLines = count($lines);
        $lines = array_slice($lines, $offset, $perPage);
        
        $logs = collect($lines)->map(function ($line, $index) use ($offset) {
            return $this->parseLogLine($line, $offset + $index + 1);
        })->filter();

        return $logs;
    }

    /**
     * Parse a single log line
     */
    private function parseLogLine($line, $lineNumber)
    {
        if (empty(trim($line))) {
            return null;
        }

        // Clean the line to prevent JavaScript issues
        $cleanLine = htmlspecialchars($line, ENT_QUOTES, 'UTF-8');

        // Laravel log format: [timestamp] environment.level: message
        $pattern = '/^\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\] (\w+)\.(\w+): (.+)$/';
        
        if (preg_match($pattern, $line, $matches)) {
            return [
                'line_number' => $lineNumber,
                'timestamp' => $matches[1],
                'environment' => $matches[2],
                'level' => strtoupper($matches[3]),
                'message' => htmlspecialchars($matches[4], ENT_QUOTES, 'UTF-8'),
                'raw' => $cleanLine
            ];
        }

        // Fallback for non-standard log formats
        return [
            'line_number' => $lineNumber,
            'timestamp' => null,
            'environment' => null,
            'level' => 'INFO',
            'message' => $cleanLine,
            'raw' => $cleanLine
        ];
    }

    /**
     * Get pagination data
     */
    private function getPagination($filename, Request $request)
    {
        $logPath = storage_path('logs/' . $filename);
        
        if (!File::exists($logPath)) {
            return null;
        }

        $content = File::get($logPath);
        $lines = explode("\n", $content);
        
        // Apply same filters as getLogEntries
        if ($request->filled('level')) {
            $lines = array_filter($lines, function ($line) use ($request) {
                return strpos($line, '[' . strtoupper($request->level) . ']') !== false;
            });
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $lines = array_filter($lines, function ($line) use ($search) {
                return stripos($line, $search) !== false;
            });
        }

        $totalLines = count($lines);
        $perPage = 50;
        $currentPage = $request->get('page', 1);
        $lastPage = ceil($totalLines / $perPage);

        return [
            'current_page' => $currentPage,
            'last_page' => $lastPage,
            'per_page' => $perPage,
            'total' => $totalLines,
            'from' => (($currentPage - 1) * $perPage) + 1,
            'to' => min($currentPage * $perPage, $totalLines),
            'has_more_pages' => $currentPage < $lastPage
        ];
    }

    /**
     * Format bytes to human readable format
     */
    private function formatBytes($bytes, $precision = 2)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }

    /**
     * Get log statistics
     */
    public function statistics()
    {
        $logFiles = $this->getLogFiles();
        $totalSize = $logFiles->sum('size');
        
        $stats = [
            'total_files' => $logFiles->count(),
            'total_size' => $this->formatBytes($logFiles->sum(function ($file) {
                return File::size($file['path']);
            })),
            'largest_file' => $logFiles->first(),
            'recent_files' => $logFiles->take(5),
            'log_levels' => $this->getLogLevelStats(),
        ];

        return view('admin.logs.statistics', compact('stats'));
    }

    /**
     * Get log level statistics
     */
    private function getLogLevelStats()
    {
        $logPath = storage_path('logs');
        $files = File::files($logPath);
        $levels = [];

        foreach ($files as $file) {
            $content = File::get($file->getPathname());
            $lines = explode("\n", $content);
            
            foreach ($lines as $line) {
                if (preg_match('/\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\] \w+\.(\w+):/', $line, $matches)) {
                    $level = strtoupper($matches[2]);
                    $levels[$level] = ($levels[$level] ?? 0) + 1;
                }
            }
        }

        return $levels;
    }
}