<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\Admin\AdminSettingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class SettingController extends Controller
{
    protected $settingService;

    public function __construct(AdminSettingService $settingService)
    {
        $this->settingService = $settingService;
    }

    /**
     * Display the settings dashboard
     */
    public function index()
    {
        $settings = $this->settingService->getAllSettings();
        $timezones = $this->settingService->getAvailableTimezones();
        $countries = $this->settingService->getAvailableCountries();

        return view('admin.settings.index', compact(
            'settings',
            'timezones',
            'countries'
        ));
    }

    /**
     * Display bank settings
     */
    public function bank()
    {
        $settings = $this->settingService->getBankSettings();

        return view('admin.settings.bank', compact('settings'));
    }

    /**
     * Update bank settings
     */
    public function updateBank(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'bank_name' => 'required|string|max:255',
            'bank_code' => 'required|string|max:10',
            'swift_code' => 'required|string|max:11',
            'bank_address' => 'required|string|max:500',
            'phone_number' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'website' => 'required|url|max:255',
            'support_email' => 'required|email|max:255',
            'compliance_email' => 'required|email|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,svg|max:2048',
            'favicon' => 'nullable|image|mimes:ico,png,jpg|max:1024',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $this->settingService->updateBankSettings($request);
            return redirect()->back()->with('success', 'Bank settings updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update bank settings: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display transaction limits settings
     */
    public function limits()
    {
        $settings = $this->settingService->getTransactionLimits();

        return view('admin.settings.limits', compact('settings'));
    }

    /**
     * Update transaction limits
     */
    public function updateLimits(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'daily_transfer_limit' => 'required|numeric|min:0',
            'monthly_transfer_limit' => 'required|numeric|min:0',
            'daily_withdrawal_limit' => 'required|numeric|min:0',
            'monthly_withdrawal_limit' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $this->settingService->updateTransactionLimits($request->all());
            return redirect()->back()->with('success', 'Transaction limits updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update transaction limits: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display compliance settings
     */
    public function compliance()
    {
        $settings = $this->settingService->getComplianceSettings();
        $countries = $this->settingService->getAvailableCountries();
        $highRiskCountries = $this->settingService->getHighRiskCountries();
        $sanctionedCountries = $this->settingService->getSanctionedCountries();

        return view('admin.settings.compliance', compact(
            'settings',
            'countries',
            'highRiskCountries',
            'sanctionedCountries'
        ));
    }

    /**
     * Update compliance settings
     */
    public function updateCompliance(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'kyc_required' => 'boolean',
            'kyc_verification_required' => 'boolean',
            'aml_monitoring' => 'boolean',
            'transaction_monitoring' => 'boolean',
            'suspicious_activity_threshold' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $this->settingService->updateComplianceSettings($request->all());
            return redirect()->back()->with('success', 'Compliance settings updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update compliance settings: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display notification settings
     */
    public function notifications()
    {
        $settings = $this->settingService->getNotificationSettings();

        return view('admin.settings.notifications', compact('settings'));
    }

    /**
     * Update notification settings
     */
    public function updateNotifications(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email_verification_enabled' => 'required|boolean',
            'smtp_driver' => 'required|string|max:50',
            'smtp_host' => 'required|string|max:255',
            'smtp_port' => 'required|integer|min:1|max:65535',
            'smtp_username' => 'required|string|max:255',
            'smtp_password' => 'required|string|max:255',
            'smtp_encryption' => 'nullable|string|max:10',
            'from_email' => 'required|email|max:255',
            'from_name' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $this->settingService->updateNotificationSettings($request->all());
            return redirect()->back()->with('success', 'Notification settings updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update notification settings: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Test SMTP configuration
     */
    public function testSmtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'smtp_driver' => 'required|string|max:50',
            'smtp_host' => 'required|string|max:255',
            'smtp_port' => 'required|integer|min:1|max:65535',
            'smtp_username' => 'required|string|max:255',
            'smtp_password' => 'required|string|max:255',
            'smtp_encryption' => 'nullable|string|max:10',
            'from_email' => 'required|email|max:255',
            'from_name' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed: ' . implode(', ', $validator->errors()->all())
            ], 400);
        }

        try {
            $result = $this->settingService->testSmtpConfiguration($request->all());
            
            return response()->json([
                'success' => $result['success'],
                'message' => $result['message']
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'SMTP test failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display system settings
     */
    public function system()
    {
        $settings = $this->settingService->getSystemSettings();

        return view('admin.settings.system', compact('settings'));
    }

    /**
     * Update system settings
     */
    public function updateSystem(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'maintenance_mode' => 'boolean',
            'debug_mode' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $this->settingService->updateSystemSettings($request->all());
            return redirect()->back()->with('success', 'System settings updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update system settings: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Run optimize:clear command
     */
    public function optimizeClear()
    {
        try {
            \Artisan::call('optimize:clear');
            $output = \Artisan::output();
            
            return response()->json([
                'success' => true,
                'message' => 'All caches cleared and application optimized successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to run optimize:clear: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Run storage:link command
     */
    public function storageLink()
    {
        try {
            \Artisan::call('storage:link');
            $output = \Artisan::output();
            
            return response()->json([
                'success' => true,
                'message' => 'Storage symbolic links created successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to run storage:link: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Test email configuration
     */
    public function testEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'smtp_host' => 'required|string|max:255',
            'smtp_port' => 'required|integer|min:1|max:65535',
            'smtp_username' => 'required|string|max:255',
            'smtp_password' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid email configuration parameters.'
            ], 400);
        }

        $result = $this->settingService->testEmailConfiguration($request->all());
        
        return response()->json($result);
    }

    /**
     * Test SMS configuration
     */
    public function testSms(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'sms_provider' => 'required|string|max:50',
            'sms_api_key' => 'required|string|max:255',
            'sms_api_secret' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid SMS configuration parameters.'
            ], 400);
        }

        $result = $this->settingService->testSmsConfiguration($request->all());
        
        return response()->json($result);
    }

    /**
     * Reset settings to default
     */
    public function reset(Request $request)
    {
        $type = $request->get('type', 'all');
        
        try {
            switch ($type) {
                case 'bank':
                    $this->settingService->updateBankSettings([]);
                    break;
                case 'limits':
                    $this->settingService->updateTransactionLimits([]);
                    break;
                case 'compliance':
                    $this->settingService->updateComplianceSettings([]);
                    break;
                case 'notifications':
                    $this->settingService->updateNotificationSettings([]);
                    break;
                case 'system':
                    $this->settingService->updateSystemSettings([]);
                    break;
                default:
                    // Reset all settings
                    $this->settingService->updateBankSettings([]);
                    $this->settingService->updateTransactionLimits([]);
                    $this->settingService->updateComplianceSettings([]);
                    $this->settingService->updateNotificationSettings([]);
                    $this->settingService->updateSystemSettings([]);
                    break;
            }

            return redirect()->back()->with('success', ucfirst($type) . ' settings reset to default values.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to reset settings: ' . $e->getMessage());
        }
    }
}
