<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\Admin\AdminReportService;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ReportController extends Controller
{
    protected $reportService;

    public function __construct(AdminReportService $reportService)
    {
        $this->reportService = $reportService;
    }

    /**
     * Display the reports index page
     */
    public function index()
    {
        $summary = $this->reportService->getDashboardSummary();
        
        return view('admin.reports.index', compact('summary'));
    }

    /**
     * Display user reports
     */
    public function users(Request $request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));
        $groupBy = $request->get('group_by', 'day');

        $registrationReport = $this->reportService->getUserRegistrationReport($startDate, $endDate, $groupBy);
        $activityReport = $this->reportService->getUserActivityReport($startDate, $endDate);
        $statusReport = $this->reportService->getUserStatusReport();

        return view('admin.reports.users', compact(
            'registrationReport', 
            'activityReport', 
            'statusReport',
            'startDate',
            'endDate',
            'groupBy'
        ));
    }

    /**
     * Display account reports
     */
    public function accounts(Request $request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));
        $groupBy = $request->get('group_by', 'day');

        $creationReport = $this->reportService->getAccountCreationReport($startDate, $endDate, $groupBy);
        $statusReport = $this->reportService->getAccountStatusReport();

        return view('admin.reports.accounts', compact(
            'creationReport',
            'statusReport',
            'startDate',
            'endDate',
            'groupBy'
        ));
    }

    /**
     * Display transaction reports
     */
    public function transactions(Request $request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));
        $groupBy = $request->get('group_by', 'day');

        $volumeReport = $this->reportService->getTransactionVolumeReport($startDate, $endDate, $groupBy);
        $typeReport = $this->reportService->getTransactionTypeReport($startDate, $endDate);
        $statusReport = $this->reportService->getTransactionStatusReport($startDate, $endDate);

        return view('admin.reports.transactions', compact(
            'volumeReport',
            'typeReport',
            'statusReport',
            'startDate',
            'endDate',
            'groupBy'
        ));
    }

    /**
     * Display KYC reports
     */
    public function kyc(Request $request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));
        $groupBy = $request->get('group_by', 'day');

        $submissionReport = $this->reportService->getKycSubmissionReport($startDate, $endDate, $groupBy);
        $statusReport = $this->reportService->getKycStatusReport();

        return view('admin.reports.kyc', compact(
            'submissionReport',
            'statusReport',
            'startDate',
            'endDate',
            'groupBy'
        ));
    }

    /**
     * Display compliance reports
     */
    public function compliance(Request $request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));
        $groupBy = $request->get('group_by', 'day');

        $flagReport = $this->reportService->getComplianceFlagReport($startDate, $endDate, $groupBy);
        $statusReport = $this->reportService->getComplianceStatusReport();
        $typeReport = $this->reportService->getComplianceTypeReport($startDate, $endDate);

        return view('admin.reports.compliance', compact(
            'flagReport',
            'statusReport',
            'typeReport',
            'startDate',
            'endDate',
            'groupBy'
        ));
    }

    /**
     * Export user data to CSV
     */
    public function exportUsers(Request $request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $users = \App\Models\User::with('profile')
            ->whereDate('created_at', '>=', $startDate)
            ->whereDate('created_at', '<=', $endDate)
            ->get()
            ->map(function($user) {
                return [
                    'ID' => $user->id,
                    'Name' => $user->name,
                    'Email' => $user->email,
                    'Customer ID' => $user->customer_id,
                    'Status' => $user->is_active ? 'Active' : 'Inactive',
                    'KYC Status' => $user->profile?->kyc_status ?? 'N/A',
                    'Created At' => $user->created_at->format('Y-m-d H:i:s'),
                ];
            })
            ->toArray();

        $filename = "users_report_{$startDate}_to_{$endDate}.csv";
        return $this->reportService->exportToCsv($users, $filename);
    }

    /**
     * Export account data to CSV
     */
    public function exportAccounts(Request $request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $accounts = \App\Models\Account::with('user')
            ->whereDate('created_at', '>=', $startDate)
            ->whereDate('created_at', '<=', $endDate)
            ->get()
            ->map(function($account) {
                return [
                    'ID' => $account->id,
                    'Account Number' => $account->account_number,
                    'Account Type' => $account->account_type,
                    'Status' => $account->status,
                    'Balance' => $account->balance,
                    'Currency' => $account->currency,
                    'User' => $account->user?->name ?? 'N/A',
                    'Created At' => $account->created_at->format('Y-m-d H:i:s'),
                ];
            })
            ->toArray();

        $filename = "accounts_report_{$startDate}_to_{$endDate}.csv";
        return $this->reportService->exportToCsv($accounts, $filename);
    }

    /**
     * Export transaction data to CSV
     */
    public function exportTransactions(Request $request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $transactions = \App\Models\Transaction::with(['user', 'fromAccount', 'toAccount'])
            ->whereDate('created_at', '>=', $startDate)
            ->whereDate('created_at', '<=', $endDate)
            ->get()
            ->map(function($transaction) {
                return [
                    'ID' => $transaction->id,
                    'Reference' => $transaction->reference_number,
                    'Type' => $transaction->type,
                    'Status' => $transaction->status,
                    'Amount' => $transaction->amount,
                    'Currency' => $transaction->currency,
                    'From Account' => $transaction->fromAccount?->account_number ?? 'N/A',
                    'To Account' => $transaction->toAccount?->account_number ?? 'N/A',
                    'User' => $transaction->user?->name ?? 'N/A',
                    'Created At' => $transaction->created_at->format('Y-m-d H:i:s'),
                ];
            })
            ->toArray();

        $filename = "transactions_report_{$startDate}_to_{$endDate}.csv";
        return $this->reportService->exportToCsv($transactions, $filename);
    }

    /**
     * Export KYC data to CSV
     */
    public function exportKyc(Request $request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $kycDocuments = \App\Models\KycDocument::with('user')
            ->whereDate('created_at', '>=', $startDate)
            ->whereDate('created_at', '<=', $endDate)
            ->get()
            ->map(function($document) {
                return [
                    'ID' => $document->id,
                    'Document Type' => $document->document_type,
                    'Status' => $document->status,
                    'Verification Level' => $document->verification_level,
                    'User' => $document->user?->name ?? 'N/A',
                    'Submitted At' => $document->created_at->format('Y-m-d H:i:s'),
                    'Reviewed At' => $document->reviewed_at?->format('Y-m-d H:i:s') ?? 'N/A',
                ];
            })
            ->toArray();

        $filename = "kyc_report_{$startDate}_to_{$endDate}.csv";
        return $this->reportService->exportToCsv($kycDocuments, $filename);
    }

    /**
     * Export compliance data to CSV
     */
    public function exportCompliance(Request $request)
    {
        $startDate = $request->get('start_date', now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $complianceLogs = \App\Models\ComplianceLog::with(['user', 'reviewer'])
            ->whereDate('created_at', '>=', $startDate)
            ->whereDate('created_at', '<=', $endDate)
            ->get()
            ->map(function($log) {
                return [
                    'ID' => $log->id,
                    'Type' => $log->type,
                    'Category' => $log->category,
                    'Severity' => $log->severity,
                    'Status' => $log->status,
                    'Description' => $log->description,
                    'User' => $log->user?->name ?? 'N/A',
                    'Reviewed By' => $log->reviewer?->name ?? 'N/A',
                    'Created At' => $log->created_at->format('Y-m-d H:i:s'),
                    'Reviewed At' => $log->reviewed_at?->format('Y-m-d H:i:s') ?? 'N/A',
                ];
            })
            ->toArray();

        $filename = "compliance_report_{$startDate}_to_{$endDate}.csv";
        return $this->reportService->exportToCsv($complianceLogs, $filename);
    }
}
