<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\User;
use App\Services\TransactionService;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

class DepositController extends Controller
{
    protected $transactionService;

    public function __construct(TransactionService $transactionService)
    {
        $this->transactionService = $transactionService;
    }

    /**
     * Display a listing of deposits.
     */
    public function index(Request $request)
    {
        $query = Transaction::with(['user', 'toAccount'])
            ->where('type', 'deposit');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('transaction_id', 'like', "%{$search}%")
                  ->orWhere('reference_number', 'like', "%{$search}%")
                  ->orWhere('external_reference', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('user', function ($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                  })
                  ->orWhereHas('toAccount', function ($accountQuery) use ($search) {
                      $accountQuery->where('account_number', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('subcategory')) {
            $query->where('subcategory', $request->subcategory);
        }

        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        if ($request->filled('currency')) {
            $query->where('currency', $request->currency);
        }

        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->filled('to_account_id')) {
            $query->where('to_account_id', $request->to_account_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->filled('amount_min')) {
            $query->where('amount', '>=', $request->amount_min);
        }

        if ($request->filled('amount_max')) {
            $query->where('amount', '<=', $request->amount_max);
        }

        // Apply sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        
        $allowedSortFields = ['created_at', 'transaction_id', 'amount', 'status', 'subcategory', 'payment_method', 'processed_at', 'completed_at'];
        if (in_array($sortBy, $allowedSortFields)) {
            $query->orderBy($sortBy, $sortDirection);
        }

        $deposits = $query->paginate(10)->withQueryString();

        // Get filter options
        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'account_name')->get();
        
        return view('admin.deposits.index', compact('deposits', 'users', 'accounts'));
    }

    /**
     * Show the form for creating a new deposit.
     */
    public function create()
    {
        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'account_name')->get();
        
        return view('admin.deposits.create', compact('users', 'accounts'));
    }

    /**
     * Store a newly created deposit.
     */
    public function store(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'to_account_id' => 'required|exists:accounts,id',
            'subcategory' => 'required|in:cheque_deposit,wire_deposit,ach_deposit,card_deposit,cash_deposit,digital_wallet_deposit,interest_deposit,refund_deposit',
            'amount' => 'required|numeric|min:0.01',
            'currency' => 'required|string|size:3',
            'exchange_rate' => 'nullable|numeric|min:0',
            'converted_amount' => 'nullable|numeric|min:0',
            'fee_amount' => 'nullable|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'net_amount' => 'nullable|numeric|min:0',
            'description' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'status' => 'required|in:pending,processing,completed,failed,cancelled,reversed',
            'verification_status' => 'required|in:pending,verified,rejected',
            'payment_method' => 'required|in:cheque,card,bank_transfer,digital_wallet,cash,wire,ach',
            'external_reference' => 'nullable|string|max:255',
            'ip_address' => 'nullable|ip',
            // Cheque specific fields
            'cheque_number' => 'nullable|string|max:50',
            'bank_name' => 'nullable|string|max:255',
            'routing_number' => 'nullable|string|max:20',
            'account_number' => 'nullable|string|max:20',
            // Card specific fields
            'card_last_four' => 'nullable|string|max:4',
            'card_type' => 'nullable|string|max:50',
            // Wire specific fields
            'wire_reference' => 'nullable|string|max:255',
            'wire_instructions' => 'nullable|string|max:1000',
        ]);

        try {
            // Prepare deposit data
            $depositData = $request->all();
            $depositData['type'] = 'deposit';
            $depositData['category'] = 'banking';

            // Add metadata for specific deposit types
            $metadata = [];
            if ($request->subcategory === 'cheque_deposit') {
                $metadata['cheque_details'] = [
                    'cheque_number' => $request->cheque_number,
                    'bank_name' => $request->bank_name,
                    'routing_number' => $request->routing_number,
                    'account_number' => $request->account_number,
                ];
            } elseif ($request->subcategory === 'card_deposit') {
                $metadata['card_details'] = [
                    'card_last_four' => $request->card_last_four,
                    'card_type' => $request->card_type,
                ];
            } elseif ($request->subcategory === 'wire_deposit') {
                $metadata['wire_details'] = [
                    'wire_reference' => $request->wire_reference,
                    'wire_instructions' => $request->wire_instructions,
                ];
            }

            if (!empty($metadata)) {
                $depositData['metadata'] = $metadata;
            }

            // Validate deposit data using service
            $validationErrors = $this->transactionService->validateTransactionData($depositData);
            if (!empty($validationErrors)) {
                return redirect()->back()
                    ->withErrors(['validation' => $validationErrors])
                    ->withInput();
            }

            // Create deposit using service
            $deposit = $this->transactionService->createTransaction($depositData);

            // Log the deposit creation
            activity()
                ->performedOn($deposit)
                ->log("Deposit created: {$deposit->transaction_id} - {$deposit->subcategory}");

            return redirect()->route('admin.deposits.show', $deposit)
                ->with('success', 'Deposit created successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create deposit: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified deposit.
     */
    public function show(Transaction $deposit)
    {
        // Ensure this is a deposit transaction
        if ($deposit->type !== 'deposit') {
            abort(404, 'Deposit not found.');
        }

        $deposit->load(['user', 'toAccount']);
        
        return view('admin.deposits.show', compact('deposit'));
    }

    /**
     * Show the form for editing the specified deposit.
     */
    public function edit(Transaction $deposit)
    {
        // Ensure this is a deposit transaction
        if ($deposit->type !== 'deposit') {
            abort(404, 'Deposit not found.');
        }

        $users = User::select('id', 'name', 'email')->get();
        $accounts = Account::select('id', 'account_number', 'account_name')->get();
        
        return view('admin.deposits.edit', compact('deposit', 'users', 'accounts'));
    }

    /**
     * Update the specified deposit.
     */
    public function update(Request $request, Transaction $deposit)
    {
        // Ensure this is a deposit transaction
        if ($deposit->type !== 'deposit') {
            abort(404, 'Deposit not found.');
        }

        $request->validate([
            'user_id' => 'required|exists:users,id',
            'to_account_id' => 'required|exists:accounts,id',
            'subcategory' => 'required|in:cheque_deposit,wire_deposit,ach_deposit,card_deposit,cash_deposit,digital_wallet_deposit,interest_deposit,refund_deposit',
            'amount' => 'required|numeric|min:0.01',
            'currency' => 'required|string|size:3',
            'exchange_rate' => 'nullable|numeric|min:0',
            'converted_amount' => 'nullable|numeric|min:0',
            'fee_amount' => 'nullable|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'net_amount' => 'nullable|numeric|min:0',
            'description' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'status' => 'required|in:pending,processing,completed,failed,cancelled,reversed',
            'verification_status' => 'required|in:pending,verified,rejected',
            'payment_method' => 'required|in:cheque,card,bank_transfer,digital_wallet,cash,wire,ach',
            'external_reference' => 'nullable|string|max:255',
            'ip_address' => 'nullable|ip',
            // Cheque specific fields
            'cheque_number' => 'nullable|string|max:50',
            'bank_name' => 'nullable|string|max:255',
            'routing_number' => 'nullable|string|max:20',
            'account_number' => 'nullable|string|max:20',
            // Card specific fields
            'card_last_four' => 'nullable|string|max:4',
            'card_type' => 'nullable|string|max:50',
            // Wire specific fields
            'wire_reference' => 'nullable|string|max:255',
            'wire_instructions' => 'nullable|string|max:1000',
        ]);

        try {
            // Prepare deposit data
            $depositData = $request->all();
            $depositData['type'] = 'deposit';
            $depositData['category'] = 'banking';

            // Add metadata for specific deposit types
            $metadata = $deposit->metadata ?? [];
            if ($request->subcategory === 'cheque_deposit') {
                $metadata['cheque_details'] = [
                    'cheque_number' => $request->cheque_number,
                    'bank_name' => $request->bank_name,
                    'routing_number' => $request->routing_number,
                    'account_number' => $request->account_number,
                ];
            } elseif ($request->subcategory === 'card_deposit') {
                $metadata['card_details'] = [
                    'card_last_four' => $request->card_last_four,
                    'card_type' => $request->card_type,
                ];
            } elseif ($request->subcategory === 'wire_deposit') {
                $metadata['wire_details'] = [
                    'wire_reference' => $request->wire_reference,
                    'wire_instructions' => $request->wire_instructions,
                ];
            }

            if (!empty($metadata)) {
                $depositData['metadata'] = $metadata;
            }

            // Update deposit using service
            $this->transactionService->updateTransaction($deposit, $depositData);

            // Log the deposit update
            activity()
                ->performedOn($deposit)
                ->log("Deposit updated: {$deposit->transaction_id} - {$deposit->subcategory}");

            return redirect()->route('admin.deposits.show', $deposit)
                ->with('success', 'Deposit updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update deposit: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified deposit from storage.
     */
    public function destroy(Transaction $deposit)
    {
        // Ensure this is a deposit transaction
        if ($deposit->type !== 'deposit') {
            abort(404, 'Deposit not found.');
        }

        // Check if deposit is completed
        if ($deposit->status === 'completed') {
            return redirect()->back()
                ->with('error', 'Cannot delete completed deposits. Please reverse the deposit instead.');
        }

        $depositId = $deposit->transaction_id;
        $deposit->delete();

        // Log the deposit deletion
        activity()
            ->log("Deposit deleted: {$depositId}");

        return redirect()->route('admin.deposits.index')
            ->with('success', 'Deposit deleted successfully.');
    }

    /**
     * Approve a deposit.
     */
    public function approve(Transaction $deposit)
    {
        // Ensure this is a deposit transaction
        if ($deposit->type !== 'deposit') {
            abort(404, 'Deposit not found.');
        }

        try {
            $this->transactionService->approveTransaction($deposit);

            activity()
                ->performedOn($deposit)
                ->log("Deposit approved: {$deposit->transaction_id}");

            return redirect()->back()
                ->with('success', 'Deposit approved successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to approve deposit: ' . $e->getMessage());
        }
    }

    /**
     * Reject a deposit.
     */
    public function reject(Transaction $deposit)
    {
        // Ensure this is a deposit transaction
        if ($deposit->type !== 'deposit') {
            abort(404, 'Deposit not found.');
        }

        try {
            $this->transactionService->rejectTransaction($deposit, 'Admin rejection');

            activity()
                ->performedOn($deposit)
                ->log("Deposit rejected: {$deposit->transaction_id}");

            return redirect()->back()
                ->with('success', 'Deposit rejected successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to reject deposit: ' . $e->getMessage());
        }
    }

    /**
     * Reverse a deposit.
     */
    public function reverse(Transaction $deposit)
    {
        // Ensure this is a deposit transaction
        if ($deposit->type !== 'deposit') {
            abort(404, 'Deposit not found.');
        }

        try {
            $this->transactionService->reverseTransaction($deposit, 'Admin reversal');

            activity()
                ->performedOn($deposit)
                ->log("Deposit reversed: {$deposit->transaction_id}");

            return redirect()->back()
                ->with('success', 'Deposit reversed successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to reverse deposit: ' . $e->getMessage());
        }
    }

    /**
     * Get deposit statistics.
     */
    public function statistics(Request $request)
    {
        $query = Transaction::where('type', 'deposit');

        // Apply date filters
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $totalDeposits = $query->count();
        $totalAmount = $query->sum('amount');
        $averageAmount = $totalDeposits > 0 ? $totalAmount / $totalDeposits : 0;

        $statusBreakdown = $query->selectRaw('status, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('status')
            ->get()
            ->keyBy('status');

        $subcategoryBreakdown = $query->selectRaw('subcategory, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('subcategory')
            ->get()
            ->keyBy('subcategory');

        $paymentMethodBreakdown = $query->selectRaw('payment_method, COUNT(*) as count, SUM(amount) as total_amount')
            ->groupBy('payment_method')
            ->get()
            ->keyBy('payment_method');

        return response()->json([
            'total_deposits' => $totalDeposits,
            'total_amount' => $totalAmount,
            'average_amount' => $averageAmount,
            'status_breakdown' => $statusBreakdown,
            'subcategory_breakdown' => $subcategoryBreakdown,
            'payment_method_breakdown' => $paymentMethodBreakdown,
        ]);
    }
}
