<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Account;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;

class CreditDebitController extends Controller
{
    /**
     * Display the credit/debit management page.
     */
    public function index(): View
    {
        $accounts = Account::with(['user'])
            ->where('status', 'active')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.credit-debit.index', compact('accounts'));
    }

    /**
     * Process credit/debit transaction for an account.
     */
    public function transaction(Request $request, Account $account): RedirectResponse
    {
        $request->validate([
            'type' => 'required|in:credit,debit',
            'amount' => 'required|numeric|min:0.01',
            'create_transaction' => 'nullable|boolean',
        ]);

        // Check if account is active
        if ($account->status !== 'active') {
            return redirect()->back()
                ->with('error', 'Cannot process transactions on inactive accounts.');
        }

        $amount = $request->amount;
        $type = $request->type;
        $createTransaction = $request->has('create_transaction');
        $description = 'Transaction';

        // Check sufficient balance for debit
        if ($type === 'debit' && $account->available_balance < $amount) {
            return redirect()->back()
                ->with('error', 'Insufficient balance. Available: ' . $account->currency_symbol . number_format($account->available_balance, 2));
        }

        try {
            // Update account balance
            if ($type === 'credit') {
                $account->increment('balance', $amount);
                $account->increment('available_balance', $amount);
                $transactionType = 'credit';
                $transactionDescription = "Credit {$description}";
            } else {
                $account->decrement('balance', $amount);
                $account->decrement('available_balance', $amount);
                $transactionType = 'debit';
                $transactionDescription = "Debit {$description}";
            }

            // Create formal transaction record if checkbox is checked
            if ($createTransaction) {
                $transaction = \App\Models\Transaction::create([
                    'transaction_id' => 'TXN' . strtoupper(uniqid()),
                    'user_id' => $account->user_id,
                    'from_account_id' => $type === 'debit' ? $account->id : null,
                    'to_account_id' => $type === 'credit' ? $account->id : null,
                    'type' => $type === 'credit' ? 'deposit' : 'withdrawal',
                    'category' => 'banking',
                    'amount' => $amount,
                    'currency' => $account->currency,
                    'description' => $transactionDescription,
                    'status' => 'completed',
                    'verification_status' => 'verified',
                    'processed_at' => now(),
                    'completed_at' => now(),
                ]);

                // Log the transaction creation
                activity()
                    ->performedOn($transaction)
                    ->log("Credit/Debit transaction created: {$transaction->transaction_id} - {$transactionDescription}");
            }

            // Log the account balance change
            activity()
                ->performedOn($account)
                ->log("Account {$type}: {$account->currency_symbol}" . number_format($amount, 2) . " - {$description}");

            $action = $type === 'credit' ? 'credited' : 'debited';
            $message = "Account successfully {$action} with {$account->currency_symbol}" . number_format($amount, 2);
            
            if ($createTransaction) {
                $message .= " and transaction record created.";
            }

            return redirect()->back()
                ->with('success', $message);

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Transaction failed: ' . $e->getMessage());
        }
    }
}
