<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ComplianceLog;
use App\Models\User;
use App\Models\Account;
use App\Models\Transaction;
use App\Models\KycDocument;
use App\Services\ComplianceService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ComplianceController extends Controller
{
    protected $complianceService;

    public function __construct(ComplianceService $complianceService)
    {
        $this->complianceService = $complianceService;
    }

    /**
     * Display a listing of compliance logs.
     */
    public function index(Request $request)
    {
        $query = ComplianceLog::with(['user', 'account', 'transaction', 'kycDocument', 'reviewer']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by severity
        if ($request->filled('severity')) {
            $query->where('severity', $request->severity);
        }

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by category
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        // Search by user name or email
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Sort
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $logs = $query->paginate(20)->withQueryString();

        // Get filter options
        $statuses = ComplianceLog::select('status')->distinct()->pluck('status');
        $severities = ComplianceLog::select('severity')->distinct()->pluck('severity');
        $types = ComplianceLog::select('type')->distinct()->pluck('type');
        $categories = ComplianceLog::select('category')->distinct()->pluck('category');

        return view('admin.compliance.index', compact(
            'logs', 'statuses', 'severities', 'types', 'categories'
        ));
    }

    /**
     * Display the specified compliance log.
     */
    public function show(ComplianceLog $compliance)
    {
        $compliance->load(['user', 'account', 'transaction', 'kycDocument', 'reviewer']);

        return view('admin.compliance.show', compact('compliance'));
    }

    /**
     * Show the form for editing the specified compliance log.
     */
    public function edit(ComplianceLog $compliance)
    {
        $compliance->load(['user', 'account', 'transaction', 'kycDocument']);

        return view('admin.compliance.edit', compact('compliance'));
    }

    /**
     * Update the specified compliance log.
     */
    public function update(Request $request, ComplianceLog $compliance)
    {
        $request->validate([
            'status' => 'required|in:pending,under_review,approved,rejected,escalated',
            'admin_notes' => 'nullable|string|max:1000',
            'resolution_notes' => 'nullable|string|max:1000',
            'resolution_status' => 'nullable|in:resolved,false_positive,requires_action',
        ]);

        try {
            DB::beginTransaction();

            $compliance->update([
                'status' => $request->status,
                'admin_notes' => $request->admin_notes,
                'resolution_notes' => $request->resolution_notes,
                'resolution_status' => $request->resolution_status,
                'reviewed_by' => Auth::id(),
                'reviewed_at' => now(),
            ]);

            // Log the update
            activity()
                ->performedOn($compliance)
                ->causedBy(Auth::user())
                ->log("Compliance log {$compliance->type} updated to {$request->status}");

            DB::commit();

            return redirect()->route('admin.compliance.show', $compliance)
                ->with('success', 'Compliance log updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to update compliance log: ' . $e->getMessage());
        }
    }

    /**
     * Approve the specified compliance log.
     */
    public function approve(Request $request, ComplianceLog $compliance)
    {
        $request->validate([
            'resolution_notes' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            $compliance->update([
                'status' => 'approved',
                'resolution_notes' => $request->resolution_notes,
                'resolution_status' => 'resolved',
                'reviewed_by' => Auth::id(),
                'reviewed_at' => now(),
            ]);

            // Log the approval
            activity()
                ->performedOn($compliance)
                ->causedBy(Auth::user())
                ->log("Compliance log {$compliance->type} approved");

            DB::commit();

            return redirect()->route('admin.compliance.show', $compliance)
                ->with('success', 'Compliance log approved successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to approve compliance log: ' . $e->getMessage());
        }
    }

    /**
     * Reject the specified compliance log.
     */
    public function reject(Request $request, ComplianceLog $compliance)
    {
        $request->validate([
            'resolution_notes' => 'required|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            $compliance->update([
                'status' => 'rejected',
                'resolution_notes' => $request->resolution_notes,
                'resolution_status' => 'false_positive',
                'reviewed_by' => Auth::id(),
                'reviewed_at' => now(),
            ]);

            // Log the rejection
            activity()
                ->performedOn($compliance)
                ->causedBy(Auth::user())
                ->log("Compliance log {$compliance->type} rejected: {$request->resolution_notes}");

            DB::commit();

            return redirect()->route('admin.compliance.show', $compliance)
                ->with('success', 'Compliance log rejected successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to reject compliance log: ' . $e->getMessage());
        }
    }

    /**
     * Escalate the specified compliance log.
     */
    public function escalate(Request $request, ComplianceLog $compliance)
    {
        $request->validate([
            'admin_notes' => 'required|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            $compliance->update([
                'status' => 'escalated',
                'admin_notes' => $request->admin_notes,
                'reviewed_by' => Auth::id(),
                'reviewed_at' => now(),
            ]);

            // Log the escalation
            activity()
                ->performedOn($compliance)
                ->causedBy(Auth::user())
                ->log("Compliance log {$compliance->type} escalated: {$request->admin_notes}");

            DB::commit();

            return redirect()->route('admin.compliance.show', $compliance)
                ->with('success', 'Compliance log escalated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to escalate compliance log: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified compliance log.
     */
    public function destroy(ComplianceLog $compliance)
    {
        try {
            $compliance->delete();

            // Log the deletion
            activity()
                ->causedBy(Auth::user())
                ->log("Compliance log {$compliance->type} deleted");

            return redirect()->route('admin.compliance.index')
                ->with('success', 'Compliance log deleted successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete compliance log: ' . $e->getMessage());
        }
    }

    /**
     * Bulk approve compliance logs.
     */
    public function bulkApprove(Request $request)
    {
        $request->validate([
            'log_ids' => 'required|array|min:1',
            'log_ids.*' => 'exists:compliance_logs,id',
            'resolution_notes' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            $approvedCount = 0;
            foreach ($request->log_ids as $logId) {
                $log = ComplianceLog::findOrFail($logId);
                if ($log->status === 'pending' || $log->status === 'under_review') {
                    $log->update([
                        'status' => 'approved',
                        'resolution_notes' => $request->resolution_notes,
                        'resolution_status' => 'resolved',
                        'reviewed_by' => Auth::id(),
                        'reviewed_at' => now(),
                    ]);
                    $approvedCount++;
                }
            }

            // Log the bulk approval
            activity()
                ->causedBy(Auth::user())
                ->log("Bulk approved {$approvedCount} compliance logs");

            DB::commit();

            return redirect()->route('admin.compliance.index')
                ->with('success', "Successfully approved {$approvedCount} compliance logs.");

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to bulk approve logs: ' . $e->getMessage());
        }
    }

    /**
     * Display compliance statistics.
     */
    public function statistics()
    {
        $stats = $this->complianceService->getComplianceStats();

        // Additional statistics
        $stats['total_logs'] = ComplianceLog::count();
        $stats['critical_pending'] = ComplianceLog::where('severity', 'critical')
            ->where('status', 'pending')
            ->count();
        $stats['high_pending'] = ComplianceLog::where('severity', 'high')
            ->where('status', 'pending')
            ->count();

        // Type breakdown
        $typeStats = ComplianceLog::select('type', DB::raw('count(*) as count'))
            ->groupBy('type')
            ->get()
            ->pluck('count', 'type');

        // Severity breakdown
        $severityStats = ComplianceLog::select('severity', DB::raw('count(*) as count'))
            ->groupBy('severity')
            ->get()
            ->pluck('count', 'severity');

        // Status breakdown
        $statusStats = ComplianceLog::select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get()
            ->pluck('count', 'status');

        // Recent activity
        $recentLogs = ComplianceLog::with(['user', 'reviewer'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('admin.compliance.statistics', compact(
            'stats', 'typeStats', 'severityStats', 'statusStats', 'recentLogs'
        ));
    }

    /**
     * Run AML monitoring via AJAX
     */
    public function amlMonitor(Request $request)
    {
        try {
            $force = $request->boolean('force', false);
            
            // Run AML monitoring
            $results = $this->complianceService->runAutomatedMonitoring();
            
            // Get AML statistics
            $stats = $this->complianceService->getAmlStats();
            
            return response()->json([
                'success' => true,
                'results' => $results,
                'stats' => $stats,
                'message' => 'AML monitoring completed successfully'
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error running AML monitoring: ' . $e->getMessage()
            ], 500);
        }
    }
}
