<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\Admin\AdminAnalyticsService;
use Illuminate\Http\Request;

class AnalyticsController extends Controller
{
    protected $analyticsService;

    public function __construct(AdminAnalyticsService $analyticsService)
    {
        $this->analyticsService = $analyticsService;
    }

    /**
     * Display the analytics dashboard
     */
    public function index(Request $request)
    {
        $period = $request->get('period', '30d');
        
        $realTimeMetrics = $this->analyticsService->getRealTimeMetrics();
        $userActivity = $this->analyticsService->getUserActivityAnalytics($period);
        $transactionTrends = $this->analyticsService->getTransactionTrendsAnalytics($period);
        $accountPerformance = $this->analyticsService->getAccountPerformanceAnalytics();
        $kycProcessing = $this->analyticsService->getKycProcessingAnalytics();
        $complianceMonitoring = $this->analyticsService->getComplianceMonitoringAnalytics();

        return view('admin.analytics.index', compact(
            'realTimeMetrics',
            'userActivity',
            'transactionTrends',
            'accountPerformance',
            'kycProcessing',
            'complianceMonitoring',
            'period'
        ));
    }

    /**
     * Display user analytics
     */
    public function users(Request $request)
    {
        $period = $request->get('period', '30d');
        
        $userGrowth = $this->analyticsService->getUserGrowthAnalytics($period);
        $userActivity = $this->analyticsService->getUserActivityAnalytics($period);
        $userDemographics = $this->analyticsService->getUserDemographicsAnalytics();

        return view('admin.analytics.users', compact(
            'userGrowth',
            'userActivity',
            'userDemographics',
            'period'
        ));
    }

    /**
     * Display transaction analytics
     */
    public function transactions(Request $request)
    {
        $period = $request->get('period', '30d');
        
        $transactionVolume = $this->analyticsService->getTransactionVolumeAnalytics($period);
        $transactionTrends = $this->analyticsService->getTransactionTrendsAnalytics($period);

        return view('admin.analytics.transactions', compact(
            'transactionVolume',
            'transactionTrends',
            'period'
        ));
    }

    /**
     * Display account analytics
     */
    public function accounts(Request $request)
    {
        $period = $request->get('period', '30d');
        
        $accountCreation = $this->analyticsService->getAccountCreationAnalytics($period);
        $accountPerformance = $this->analyticsService->getAccountPerformanceAnalytics();

        return view('admin.analytics.accounts', compact(
            'accountCreation',
            'accountPerformance',
            'period'
        ));
    }

    /**
     * Display KYC analytics
     */
    public function kyc(Request $request)
    {
        $period = $request->get('period', '30d');
        
        $kycSubmission = $this->analyticsService->getKycSubmissionAnalytics($period);
        $kycProcessing = $this->analyticsService->getKycProcessingAnalytics();

        return view('admin.analytics.kyc', compact(
            'kycSubmission',
            'kycProcessing',
            'period'
        ));
    }

    /**
     * Display compliance analytics
     */
    public function compliance(Request $request)
    {
        $period = $request->get('period', '30d');
        
        $complianceFlags = $this->analyticsService->getComplianceFlagAnalytics($period);
        $complianceMonitoring = $this->analyticsService->getComplianceMonitoringAnalytics();

        return view('admin.analytics.compliance', compact(
            'complianceFlags',
            'complianceMonitoring',
            'period'
        ));
    }

    /**
     * Get real-time metrics via AJAX
     */
    public function realTimeMetrics()
    {
        $metrics = $this->analyticsService->getRealTimeMetrics();
        
        return response()->json($metrics);
    }

    /**
     * Get user growth data via AJAX
     */
    public function userGrowthData(Request $request)
    {
        $period = $request->get('period', '30d');
        $data = $this->analyticsService->getUserGrowthAnalytics($period);
        
        return response()->json($data);
    }

    /**
     * Get transaction volume data via AJAX
     */
    public function transactionVolumeData(Request $request)
    {
        $period = $request->get('period', '30d');
        $data = $this->analyticsService->getTransactionVolumeAnalytics($period);
        
        return response()->json($data);
    }

    /**
     * Get account creation data via AJAX
     */
    public function accountCreationData(Request $request)
    {
        $period = $request->get('period', '30d');
        $data = $this->analyticsService->getAccountCreationAnalytics($period);
        
        return response()->json($data);
    }

    /**
     * Get KYC submission data via AJAX
     */
    public function kycSubmissionData(Request $request)
    {
        $period = $request->get('period', '30d');
        $data = $this->analyticsService->getKycSubmissionAnalytics($period);
        
        return response()->json($data);
    }

    /**
     * Get compliance flag data via AJAX
     */
    public function complianceFlagData(Request $request)
    {
        $period = $request->get('period', '30d');
        $data = $this->analyticsService->getComplianceFlagAnalytics($period);
        
        return response()->json($data);
    }
}
