<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Account;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

class AccountController extends Controller
{
    /**
     * Display a listing of accounts.
     */
    public function index(Request $request)
    {
        $query = Account::with(['user', 'parentAccount']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('account_number', 'like', "%{$search}%")
                  ->orWhere('account_name', 'like', "%{$search}%")
                  ->orWhereHas('user', function ($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('account_type')) {
            $query->where('account_type', $request->account_type);
        }

        if ($request->filled('currency')) {
            $query->where('currency', $request->currency);
        }

        if ($request->filled('verification_status')) {
            $query->where('verification_status', $request->verification_status);
        }

        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->filled('is_master_account')) {
            $query->where('is_master_account', $request->is_master_account === '1');
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Apply sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        
        $allowedSortFields = ['created_at', 'account_number', 'account_name', 'balance', 'status', 'account_type'];
        if (in_array($sortBy, $allowedSortFields)) {
            $query->orderBy($sortBy, $sortDirection);
        }

        $accounts = $query->paginate(10)->withQueryString();

        // Get filter options
        $users = User::select('id', 'name', 'email')->get();
        
        return view('admin.accounts.index', compact('accounts', 'users'));
    }

    /**
     * Show the form for creating a new account.
     */
    public function create()
    {
        $users = User::select('id', 'name', 'email')->get();
        $parentAccounts = Account::where('is_master_account', true)->get();
        
        return view('admin.accounts.create', compact('users', 'parentAccounts'));
    }

    /**
     * Store a newly created account.
     */
    public function store(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'account_type' => 'required|in:savings,checking,business,investment,fixed_deposit,offshore',
            'account_name' => 'required|string|max:255',
            'currency' => 'required|string|size:3',
            'description' => 'nullable|string|max:1000',
            'balance' => 'nullable|numeric|min:0',
            'available_balance' => 'nullable|numeric|min:0',
            'hold_balance' => 'nullable|numeric|min:0',
            'daily_transfer_limit' => 'nullable|numeric|min:0',
            'monthly_transfer_limit' => 'nullable|numeric|min:0',
            'daily_withdrawal_limit' => 'nullable|numeric|min:0',
            'monthly_withdrawal_limit' => 'nullable|numeric|min:0',
            'status' => 'required|in:active,inactive,suspended,frozen,closed,pending',
            'verification_status' => 'required|in:pending,verified,rejected',
            'interest_rate' => 'nullable|numeric|min:0|max:1',
            'interest_frequency' => 'nullable|in:monthly,quarterly,annually',
            'monthly_maintenance_fee' => 'nullable|numeric|min:0',
            'overdraft_limit' => 'nullable|numeric|min:0',
            'overdraft_fee' => 'nullable|numeric|min:0',
            'risk_level' => 'required|in:low,medium,high',
            'risk_tolerance' => 'nullable|in:conservative,moderate,aggressive',
            'parent_account_id' => 'nullable|exists:accounts,id',
            'is_master_account' => 'boolean',
            'business_name' => 'nullable|string|max:255',
            'business_type' => 'nullable|string|max:255',
            'business_registration_number' => 'nullable|string|max:255',
            'tax_identification_number' => 'nullable|string|max:255',
            'investment_strategy' => 'nullable|string|max:1000',
            'investment_preferences' => 'nullable|array',
            
            // Boolean features
            'online_banking_enabled' => 'boolean',
            'mobile_banking_enabled' => 'boolean',
            'debit_card_enabled' => 'boolean',
            'credit_card_enabled' => 'boolean',
            'international_transfers_enabled' => 'boolean',
            'investment_trading_enabled' => 'boolean',
            'two_factor_required' => 'boolean',
            'biometric_required' => 'boolean',
            'aml_monitoring_enabled' => 'boolean',
        ]);

        // Generate unique account number
        $accountNumber = $this->generateAccountNumber($request->account_type, $request->currency);

        $account = Account::create([
            'user_id' => $request->user_id,
            'account_number' => $accountNumber,
            'account_type' => $request->account_type,
            'account_name' => $request->account_name,
            'currency' => $request->currency,
            'description' => $request->description,
            'balance' => $request->balance ?: 0,
            'available_balance' => $request->available_balance ?: $request->balance ?: 0,
            'hold_balance' => $request->hold_balance ?: 0,
            'daily_transfer_limit' => $request->daily_transfer_limit,
            'monthly_transfer_limit' => $request->monthly_transfer_limit,
            'daily_withdrawal_limit' => $request->daily_withdrawal_limit,
            'monthly_withdrawal_limit' => $request->monthly_withdrawal_limit,
            'status' => $request->status,
            'verification_status' => $request->verification_status,
            'interest_rate' => $request->interest_rate,
            'interest_frequency' => $request->interest_frequency,
            'monthly_maintenance_fee' => $request->monthly_maintenance_fee,
            'overdraft_limit' => $request->overdraft_limit,
            'overdraft_fee' => $request->overdraft_fee,
            'risk_level' => $request->risk_level,
            'risk_tolerance' => $request->risk_tolerance,
            'parent_account_id' => $request->parent_account_id,
            'is_master_account' => $request->boolean('is_master_account'),
            'business_name' => $request->business_name,
            'business_type' => $request->business_type,
            'business_registration_number' => $request->business_registration_number,
            'tax_identification_number' => $request->tax_identification_number,
            'investment_strategy' => $request->investment_strategy,
            'investment_preferences' => $request->investment_preferences,
            
            // Boolean features
            'online_banking_enabled' => $request->boolean('online_banking_enabled'),
            'mobile_banking_enabled' => $request->boolean('mobile_banking_enabled'),
            'debit_card_enabled' => $request->boolean('debit_card_enabled'),
            'credit_card_enabled' => $request->boolean('credit_card_enabled'),
            'international_transfers_enabled' => $request->boolean('international_transfers_enabled'),
            'investment_trading_enabled' => $request->boolean('investment_trading_enabled'),
            'two_factor_required' => $request->boolean('two_factor_required'),
            'biometric_required' => $request->boolean('biometric_required'),
            'aml_monitoring_enabled' => $request->boolean('aml_monitoring_enabled'),
            
            'activated_at' => $request->status === 'active' ? now() : null,
        ]);

        // Log the account creation
        activity()
            ->performedOn($account)
            ->log("Account created: {$account->account_number} for user: {$account->user->name}");

        return redirect()->route('admin.accounts.show', $account)
            ->with('success', 'Account created successfully.');
    }

    /**
     * Display the specified account.
     */
    public function show(Account $account)
    {
        $account->load(['user', 'parentAccount', 'childAccounts']);
        
        // Load recent transactions using the allTransactions method
        $account->recentTransactions = $account->allTransactions()
            ->with(['fromAccount', 'toAccount', 'user'])
            ->latest()
            ->limit(10)
            ->get();

        return view('admin.accounts.show', compact('account'));
    }

    /**
     * Show the form for editing the specified account.
     */
    public function edit(Account $account)
    {
        $users = User::select('id', 'name', 'email')->get();
        $parentAccounts = Account::where('is_master_account', true)
            ->where('id', '!=', $account->id)
            ->get();
        
        return view('admin.accounts.edit', compact('account', 'users', 'parentAccounts'));
    }

    /**
     * Update the specified account.
     */
    public function update(Request $request, Account $account)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'account_type' => 'required|in:savings,checking,business,investment,fixed_deposit,offshore',
            'account_name' => 'required|string|max:255',
            'currency' => 'required|string|size:3',
            'description' => 'nullable|string|max:1000',
            'balance' => 'nullable|numeric|min:0',
            'available_balance' => 'nullable|numeric|min:0',
            'hold_balance' => 'nullable|numeric|min:0',
            'daily_transfer_limit' => 'nullable|numeric|min:0',
            'monthly_transfer_limit' => 'nullable|numeric|min:0',
            'daily_withdrawal_limit' => 'nullable|numeric|min:0',
            'monthly_withdrawal_limit' => 'nullable|numeric|min:0',
            'status' => 'required|in:active,inactive,suspended,frozen,closed,pending',
            'verification_status' => 'required|in:pending,verified,rejected',
            'interest_rate' => 'nullable|numeric|min:0|max:1',
            'interest_frequency' => 'nullable|in:monthly,quarterly,annually',
            'monthly_maintenance_fee' => 'nullable|numeric|min:0',
            'overdraft_limit' => 'nullable|numeric|min:0',
            'overdraft_fee' => 'nullable|numeric|min:0',
            'risk_level' => 'required|in:low,medium,high',
            'risk_tolerance' => 'nullable|in:conservative,moderate,aggressive',
            'parent_account_id' => 'nullable|exists:accounts,id',
            'is_master_account' => 'boolean',
            'business_name' => 'nullable|string|max:255',
            'business_type' => 'nullable|string|max:255',
            'business_registration_number' => 'nullable|string|max:255',
            'tax_identification_number' => 'nullable|string|max:255',
            'investment_strategy' => 'nullable|string|max:1000',
            'investment_preferences' => 'nullable|array',
            
            // Boolean features
            'online_banking_enabled' => 'boolean',
            'mobile_banking_enabled' => 'boolean',
            'debit_card_enabled' => 'boolean',
            'credit_card_enabled' => 'boolean',
            'international_transfers_enabled' => 'boolean',
            'investment_trading_enabled' => 'boolean',
            'two_factor_required' => 'boolean',
            'biometric_required' => 'boolean',
            'aml_monitoring_enabled' => 'boolean',
        ]);

        $oldStatus = $account->status;

        $account->update([
            'user_id' => $request->user_id,
            'account_type' => $request->account_type,
            'account_name' => $request->account_name,
            'currency' => $request->currency,
            'description' => $request->description,
            'balance' => $request->balance ?: $account->balance,
            'available_balance' => $request->available_balance ?: $account->available_balance,
            'hold_balance' => $request->hold_balance ?: $account->hold_balance,
            'daily_transfer_limit' => $request->daily_transfer_limit,
            'monthly_transfer_limit' => $request->monthly_transfer_limit,
            'daily_withdrawal_limit' => $request->daily_withdrawal_limit,
            'monthly_withdrawal_limit' => $request->monthly_withdrawal_limit,
            'status' => $request->status,
            'verification_status' => $request->verification_status,
            'interest_rate' => $request->interest_rate,
            'interest_frequency' => $request->interest_frequency,
            'monthly_maintenance_fee' => $request->monthly_maintenance_fee,
            'overdraft_limit' => $request->overdraft_limit,
            'overdraft_fee' => $request->overdraft_fee,
            'risk_level' => $request->risk_level,
            'risk_tolerance' => $request->risk_tolerance,
            'parent_account_id' => $request->parent_account_id,
            'is_master_account' => $request->boolean('is_master_account'),
            'business_name' => $request->business_name,
            'business_type' => $request->business_type,
            'business_registration_number' => $request->business_registration_number,
            'tax_identification_number' => $request->tax_identification_number,
            'investment_strategy' => $request->investment_strategy,
            'investment_preferences' => $request->investment_preferences,
            
            // Boolean features
            'online_banking_enabled' => $request->boolean('online_banking_enabled'),
            'mobile_banking_enabled' => $request->boolean('mobile_banking_enabled'),
            'debit_card_enabled' => $request->boolean('debit_card_enabled'),
            'credit_card_enabled' => $request->boolean('credit_card_enabled'),
            'international_transfers_enabled' => $request->boolean('international_transfers_enabled'),
            'investment_trading_enabled' => $request->boolean('investment_trading_enabled'),
            'two_factor_required' => $request->boolean('two_factor_required'),
            'biometric_required' => $request->boolean('biometric_required'),
            'aml_monitoring_enabled' => $request->boolean('aml_monitoring_enabled'),
            
            'activated_at' => $request->status === 'active' && $oldStatus !== 'active' ? now() : $account->activated_at,
            'closed_at' => $request->status === 'closed' && $oldStatus !== 'closed' ? now() : $account->closed_at,
        ]);

        // Log the account update
        activity()
            ->performedOn($account)
            ->log("Account updated: {$account->account_number} - Status changed from {$oldStatus} to {$request->status}");

        return redirect()->route('admin.accounts.show', $account)
            ->with('success', 'Account updated successfully.');
    }

    /**
     * Remove the specified account from storage.
     */
    public function destroy(Account $account)
    {
        // Check if account has transactions
        if ($account->transactions()->count() > 0) {
            return redirect()->back()
                ->with('error', 'Cannot delete account with existing transactions. Please close the account instead.');
        }

        // Check if account has child accounts
        if ($account->childAccounts()->count() > 0) {
            return redirect()->back()
                ->with('error', 'Cannot delete account with child accounts. Please reassign or delete child accounts first.');
        }

        $accountNumber = $account->account_number;
        $account->delete();

        // Log the account deletion
        activity()
            ->log("Account deleted: {$accountNumber}");

        return redirect()->route('admin.accounts.index')
            ->with('success', 'Account deleted successfully.');
    }

    /**
     * Generate a unique account number.
     */
    private function generateAccountNumber(string $accountType, string $currency): string
    {
        $prefix = match($accountType) {
            'savings' => '10',
            'checking' => '20',
            'business' => '30',
            'investment' => '40',
            'fixed_deposit' => '50',
            'offshore' => '60',
            default => '00'
        };

        $currencyCode = match($currency) {
            'USD' => '01',
            'EUR' => '02',
            'GBP' => '03',
            default => '00'
        };

        do {
            $randomNumber = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
            $accountNumber = $prefix . $currencyCode . $randomNumber;
        } while (Account::where('account_number', $accountNumber)->exists());

        return $accountNumber;
    }

    /**
     * Suspend an account.
     */
    public function suspend(Account $account)
    {
        $account->update([
            'status' => 'suspended',
            'closed_at' => now()
        ]);

        activity()
            ->performedOn($account)
            ->log("Account suspended: {$account->account_number}");

        return redirect()->back()
            ->with('success', 'Account suspended successfully.');
    }

    /**
     * Activate an account.
     */
    public function activate(Account $account)
    {
        $account->update([
            'status' => 'active',
            'activated_at' => now(),
            'closed_at' => null
        ]);

        activity()
            ->performedOn($account)
            ->log("Account activated: {$account->account_number}");

        return redirect()->back()
            ->with('success', 'Account activated successfully.');
    }

    /**
     * Close an account.
     */
    public function close(Account $account)
    {
        $account->update([
            'status' => 'closed',
            'closed_at' => now()
        ]);

        activity()
            ->performedOn($account)
            ->log("Account closed: {$account->account_number}");

        return redirect()->back()
            ->with('success', 'Account closed successfully.');
    }

    /**
     * Process credit/debit transaction for an account.
     */
    public function transaction(Request $request, Account $account)
    {
        $request->validate([
            'type' => 'required|in:credit,debit',
            'amount' => 'required|numeric|min:0.01',
            'create_transaction' => 'nullable|boolean',
        ]);

        // Check if account is active
        if ($account->status !== 'active') {
            return redirect()->back()
                ->with('error', 'Cannot process transactions on inactive accounts.');
        }

        $amount = $request->amount;
        $type = $request->type;
        $createTransaction = $request->has('create_transaction');
        $description = 'Transaction';

        // Check sufficient balance for debit
        if ($type === 'debit' && $account->available_balance < $amount) {
            return redirect()->back()
                ->with('error', 'Insufficient balance. Available: ' . $account->currency_symbol . number_format($account->available_balance, 2));
        }

        try {
            // Update account balance
            if ($type === 'credit') {
                $account->increment('balance', $amount);
                $account->increment('available_balance', $amount);
                $transactionType = 'credit';
                $transactionDescription = "Credit {$description}";
            } else {
                $account->decrement('balance', $amount);
                $account->decrement('available_balance', $amount);
                $transactionType = 'debit';
                $transactionDescription = "Debit {$description}";
            }

            // Create formal transaction record if checkbox is checked
            if ($createTransaction) {
                $transaction = \App\Models\Transaction::create([
                    'transaction_id' => 'TXN' . strtoupper(uniqid()),
                    'user_id' => $account->user_id,
                    'from_account_id' => $type === 'debit' ? $account->id : null,
                    'to_account_id' => $type === 'credit' ? $account->id : null,
                    'type' => $type === 'credit' ? 'deposit' : 'withdrawal',
                    'category' => 'Banking',
                    'amount' => $amount,
                    'currency' => $account->currency,
                    'description' => $transactionDescription,
                    'status' => 'completed',
                    'verification_status' => 'verified',
                    'processed_at' => now(),
                    'completed_at' => now(),
                ]);

                // Log the transaction creation
                activity()
                    ->performedOn($transaction)
                    ->log("Admin transaction created: {$transaction->transaction_id} - {$transactionDescription}");
            }

            // Log the account balance change
            activity()
                ->performedOn($account)
                ->log("Admin {$type}: {$account->currency_symbol}" . number_format($amount, 2) . " - {$description}");

            $action = $type === 'credit' ? 'credited' : 'debited';
            $message = "Account successfully {$action} with {$account->currency_symbol}" . number_format($amount, 2);
            
            if ($createTransaction) {
                $message .= " and transaction record created.";
            }

            return redirect()->back()
                ->with('success', $message);

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Transaction failed: ' . $e->getMessage());
        }
    }
}
