<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\TransactionGeneratorService;
use App\Models\Account;
use Carbon\Carbon;

class GenerateTransactionsCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'transactions:generate 
                            {account_id : The account ID to generate transactions for}
                            {--start-date= : Start date (Y-m-d format)}
                            {--end-date= : End date (Y-m-d format)}
                            {--count=10 : Number of transactions to generate}
                            {--direction=mixed : Transaction direction (incoming, outgoing, mixed)}
                            {--all-accounts : Generate for all accounts}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate sample transactions for testing purposes';

    protected $transactionGenerator;

    public function __construct(TransactionGeneratorService $transactionGenerator)
    {
        parent::__construct();
        $this->transactionGenerator = $transactionGenerator;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 Starting transaction generation...');

        // Get parameters
        $accountId = $this->argument('account_id');
        $startDate = $this->option('start-date') ? Carbon::parse($this->option('start-date')) : Carbon::now()->subDays(30);
        $endDate = $this->option('end-date') ? Carbon::parse($this->option('end-date')) : Carbon::now();
        $count = (int) $this->option('count');
        $direction = $this->option('direction');
        $allAccounts = $this->option('all-accounts');

        // Validate direction
        if (!in_array($direction, ['incoming', 'outgoing', 'mixed'])) {
            $this->error('Invalid direction. Must be: incoming, outgoing, or mixed');
            return 1;
        }

        // Validate date range
        if ($startDate->gt($endDate)) {
            $this->error('Start date cannot be after end date');
            return 1;
        }

        try {
            if ($allAccounts) {
                // Generate for all accounts
                $accounts = Account::where('status', 'active')->get();
                
                if ($accounts->isEmpty()) {
                    $this->error('No active accounts found');
                    return 1;
                }

                $this->info("Generating {$count} transactions for {$accounts->count()} accounts...");
                $this->info("Date range: {$startDate->format('Y-m-d')} to {$endDate->format('Y-m-d')}");
                $this->info("Direction: {$direction}");

                $accountIds = $accounts->pluck('id')->toArray();
                $transactions = $this->transactionGenerator->generateBulkTransactions(
                    $accountIds,
                    $startDate,
                    $endDate,
                    $count,
                    $direction
                );

                $this->info("✅ Generated " . count($transactions) . " transactions across all accounts");
            } else {
                // Generate for specific account
                $account = Account::find($accountId);
                
                if (!$account) {
                    $this->error("Account with ID {$accountId} not found");
                    return 1;
                }

                $this->info("Generating {$count} transactions for account: {$account->account_number} ({$account->account_name})");
                $this->info("Date range: {$startDate->format('Y-m-d')} to {$endDate->format('Y-m-d')}");
                $this->info("Direction: {$direction}");

                $transactions = $this->transactionGenerator->generateTransactions(
                    $accountId,
                    $startDate,
                    $endDate,
                    $count,
                    $direction
                );

                $this->info("✅ Generated " . count($transactions) . " transactions for account {$account->account_number}");
            }

            // Show summary
            $this->showSummary($transactions);

            return 0;

        } catch (\Exception $e) {
            $this->error("Error generating transactions: " . $e->getMessage());
            return 1;
        }
    }

    /**
     * Show transaction summary
     */
    private function showSummary(array $transactions): void
    {
        $this->newLine();
        $this->info('📊 Transaction Summary:');
        
        $typeCounts = [];
        $totalAmount = 0;
        $incomingCount = 0;
        $outgoingCount = 0;

        foreach ($transactions as $transaction) {
            $type = $transaction->type;
            $typeCounts[$type] = ($typeCounts[$type] ?? 0) + 1;
            $totalAmount += $transaction->amount;

            if ($transaction->to_account_id && !$transaction->from_account_id) {
                $incomingCount++;
            } elseif ($transaction->from_account_id && !$transaction->to_account_id) {
                $outgoingCount++;
            }
        }

        $this->table(
            ['Type', 'Count', 'Percentage'],
            collect($typeCounts)->map(function ($count, $type) use ($transactions) {
                $percentage = round(($count / count($transactions)) * 100, 1);
                return [$type, $count, "{$percentage}%"];
            })->toArray()
        );

        $this->info("Total Amount: $" . number_format($totalAmount, 2));
        $this->info("Incoming Transactions: {$incomingCount}");
        $this->info("Outgoing Transactions: {$outgoingCount}");
    }
}
