<?php

namespace App\Console\Commands;

use App\Models\User;
use App\Services\NotificationService;
use Illuminate\Console\Command;

class CreateTestNotifications extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'notifications:test {user_id?}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create test notifications for demonstration purposes';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $userId = $this->argument('user_id');
        
        if ($userId) {
            $user = User::find($userId);
            if (!$user) {
                $this->error("User with ID {$userId} not found.");
                return;
            }
            if (!$user->hasRole('customer')) {
                $this->error("User with ID {$userId} does not have the 'customer' role.");
                return;
            }
            $users = collect([$user]);
        } else {
            $users = User::role('customer')->take(3)->get();
            if ($users->isEmpty()) {
                $this->error('No users with "customer" role found in the database.');
                return;
            }
        }

        $notificationService = app(NotificationService::class);

        foreach ($users as $user) {
            $this->info("Creating test notifications for user: {$user->name}");

            // Create various test notifications
            $notificationService->notifyNewUserWelcome($user);
            $notificationService->notifyLoginIpChange($user, '192.168.1.100', 'New York, NY');
            $notificationService->notifyKycApproved($user);
            $notificationService->notifyAccountCreated($user, '5001123456', 'savings', 'USD', true);
            $notificationService->notifyVirtualCardCreated($user, 'Visa Debit', '1234', [
                'card_brand' => 'visa',
                'card_type' => 'debit',
                'card_name' => $user->name . ' Card',
                'currency' => 'USD',
                'expiry_date' => '12/2027',
                'daily_limit' => 1000.00,
                'monthly_limit' => 10000.00,
                'account_number' => '5001123456',
            ]);
            $notificationService->notifyTransferCompleted($user, 500.00, 'USD', 'John Doe', 'TXN-123456');
            $notificationService->notifyInterAccountTransfer($user, 250.00, 'EUR', 'Savings Account', 'Checking Account');
            $notificationService->notifyChequeDeposit($user, 1200.00, 'USD', 'completed');
            $notificationService->notifyPasswordChanged($user);
            $notificationService->notifyPinChanged($user);
            $notificationService->notifyBeneficiaryAdded($user, 'Jane Smith', 'ACC-789012');
            $notificationService->notifyTwoFactorEnabled($user, 'email');
        }

        $this->info('Test notifications created successfully!');
        $this->info('You can now check the notification dropdown in the header.');
    }
}
